<?php


namespace Gek\Collections;


use ArrayAccess;
use Gek\Collections\Traits\TypedCollectionTrait;
use Gek\Infrastructure\Exceptions\InvalidArgumentException;
use Gek\Infrastructure\Str;

/**
 * Class TypedDictionaryWrapper
 * @package Gek\Collections
 * @method void add($item)
 * @method void addRange($items)
 * @method bool contains($item)
 * @method bool remove($item)
 * @method void addKeyValue(string $key, $item)
 * @method bool tryAddKeyValue(string $key, $item)
 * @method mixed getByKey(string $key)
 * @method void setByKey(string $key, $item)
 */
abstract class TypedDictionaryWrapper extends EnumerableWrapper implements ArrayAccess, \Serializable
{
    use TypedCollectionTrait;

    #region fields

    protected IDictionary $innerDictionary;

    #endregion fields

    #region ctor

    /**
     * TypedDictionaryWrapper constructor.
     * @param IDictionary $dictionary
     * @param string $itemType
     */
    protected function __construct(IDictionary $dictionary, string $itemType)
    {
        $this->innerDictionary = $dictionary;
        if(!$this->initalizeType($itemType)){
            throw new InvalidArgumentException(
                Str::format('$itemType ({0}) geçersiz tip adı.', $itemType),
                0,
                null,
                '$itemType'
            );
        }
        parent::__construct([$this,'getInnerDictionary']);
    }

    #endregion ctor

    #region methods

    /**
     * @return IDictionary
     */
    public function getInnerDictionary():IDictionary{
        return $this->innerDictionary;
    }

    /**
     *
     */
    public function clear(): void
    {
        $this->innerDictionary->clear();
    }

    /**
     * @param string $key
     * @return bool
     */
    public function containsKey(string $key): bool
    {
        return $this->innerDictionary->containsKey($key);
    }



    /**
     * @param string $key
     * @return bool
     */
    public function removeByKey(string $key): bool
    {
        return $this->innerDictionary->removeByKey($key);
    }

    /**
     * @return Enumerable
     */
    public function getKeys(): Enumerable
    {
        return $this->innerDictionary->getKeys();
    }

    /**
     * @return Enumerable
     */
    public function getValues(): Enumerable
    {
        return $this->innerDictionary->getValues();
    }

    #region ArrayAccess

    /**
     * Whether a offset exists
     * @link https://php.net/manual/en/arrayaccess.offsetexists.php
     * @param mixed $offset <p>
     * An offset to check for.
     * </p>
     * @return bool true on success or false on failure.
     * </p>
     * <p>
     * The return value will be casted to boolean if non-boolean was returned.
     * @since 5.0.0
     */
    public function offsetExists($offset)
    {
        return $this->innerDictionary->offsetExists($offset);
    }

    /**
     * Offset to retrieve
     * @link https://php.net/manual/en/arrayaccess.offsetget.php
     * @param mixed $offset <p>
     * The offset to retrieve.
     * </p>
     * @return mixed Can return all value types.
     * @since 5.0.0
     */
    public function offsetGet($offset)
    {
        return $this->innerDictionary->offsetGet($offset);
    }

    /**
     * Offset to set
     * @link https://php.net/manual/en/arrayaccess.offsetset.php
     * @param mixed $offset <p>
     * The offset to assign the value to.
     * </p>
     * @param mixed $value <p>
     * The value to set.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetSet($offset, $value)
    {
        if (!$this->checkItemType($value)) {
            throw new InvalidArgumentException(
                Str::format('$value ("{0}"), geçerli bir "{1}" öğesi değil.', $value, $this->itemType),
                0,
                null,
                '$value'
            );
        }
        $this->innerDictionary->offsetSet($offset,$value);

    }

    /**
     * Offset to unset
     * @link https://php.net/manual/en/arrayaccess.offsetunset.php
     * @param mixed $offset <p>
     * The offset to unset.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetUnset($offset)
    {
        $this->innerDictionary->offsetUnset($offset);
    }

    #endregion ArrayAccess

    #endregion methods

    #region Serializable
    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     */
    public function serialize()
    {
        $nonSerializableTypes = ['callable', 'closure', 'resource'];
        if(in_array(strtolower($this->itemType),$nonSerializableTypes)){
            throw new \LogicException('Öğe tipi (' . strtolower($this->itemType) . ') serileştirilebilir bir tip değil.');
        }
        $dt = [
            'id' => $this->innerDictionary,
            'it' => $this->itemType
        ];
        return serialize($dt);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     */
    public function unserialize($serialized)
    {
        $dt = unserialize($serialized);
        $this->innerDictionary = $dt['id'];
        if(!$this->initalizeType($dt['it'])){
            throw new InvalidArgumentException(
                Str::format('$dt[\'it\'] ({0}) geçersiz tip adı.', $dt['it']),
                0,
                null,
                '$dt[\'it\']'
            );
        }

        $this->getEnumFn = \Closure::fromCallable([$this,'getInnerDictionary']);
    }


    #endregion Serializable

}
