<?php


namespace Gek\Collections\Iterators;


use Closure;
use Iterator;

/**
 * Class IntersectIterator
 * @package Gek\Collections\Iterators
 */
class IntersectIterator implements Iterator
{
    #region field

    protected Iterator $iterator;

    protected Iterator $secondIterator;

    protected ?array $tempArray = null;

    protected ?Closure $comparerFn = null;

    #endregion field

    #region ctor
    /**
     * IntersectIterator constructor.
     * @param Iterator $iterator
     * @param Iterator $second
     * @param callable|null $comparerFn
     */
    public function __construct(Iterator $iterator, Iterator $second, ?callable $comparerFn = null)
    {
        $this->iterator = $iterator;
        $this->secondIterator = $second;
        if($comparerFn !== null){
            $this->comparerFn = ($comparerFn instanceof Closure) ? $comparerFn : Closure::fromCallable($comparerFn);
        }
    }

    #endregion ctor

    #region Iterator
    /**
     * Return the current element
     * @link https://php.net/manual/en/iterator.current.php
     * @return mixed Can return any type.
     * @since 5.0.0
     */
    public function current()
    {
        $this->check();
        return $this->iterator->current();
    }

    /**
     * Move forward to next element
     * @link https://php.net/manual/en/iterator.next.php
     * @return void Any returned value is ignored.
     * @since 5.0.0
     */
    public function next()
    {
        $this->check();
        $this->iterator->next();

    }

    /**
     * Return the key of the current element
     * @link https://php.net/manual/en/iterator.key.php
     * @return mixed scalar on success, or null on failure.
     * @since 5.0.0
     */
    public function key()
    {
        $this->check();
        return $this->iterator->key();
    }

    /**
     * Checks if current position is valid
     * @link https://php.net/manual/en/iterator.valid.php
     * @return bool The return value will be casted to boolean and then evaluated.
     * Returns true on success or false on failure.
     * @since 5.0.0
     */
    public function valid()
    {
        $this->check();
        if(empty($this->tempArray)){
            return false;
        }
        if($this->comparerFn === null){
            while($this->iterator->valid() && !in_array($this->iterator->current(),$this->tempArray)){
                $this->iterator->next();
            }
        }else{
            $notFound = true;
            while ($this->iterator->valid() && $notFound){
                $found = false;
                foreach ($this->tempArray as $itm){
                    $found = $this->comparerFn->call($this,$this->iterator->current(), $itm);
                    if($found == true){
                        break;
                    }
                }
                $notFound = !$found;
                if($notFound){
                    $this->iterator->next();
                }
            }
        }
        return $this->iterator->valid();
    }

    /**
     * Rewind the Iterator to the first element
     * @link https://php.net/manual/en/iterator.rewind.php
     * @return void Any returned value is ignored.
     * @since 5.0.0
     */
    public function rewind()
    {
        $this->tempArray = null;
        $this->iterator->rewind();
    }
    #endregion Iterator

    #region Utils

    protected function check(){
        if($this->tempArray !== null){
            return;
        }
        $this->tempArray = iterator_to_array($this->secondIterator);
    }

    #endregion Utils
}
