<?php


namespace Gek\Collections\Tests\Typed;


use ArrayIterator;
use ArrayObject;
use Gek\Collections\ArrayList;
use Gek\Collections\Enumerable;
use Gek\Collections\IEnumerable;
use Gek\Collections\Iterators\ArrayListIterator;
use Gek\Collections\Tests\Fixtures\MockRefObj;
use Gek\Collections\Typed\IntList;
use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Iterator;
use PHPUnit\Framework\TestCase;
use TypeError;

/**
 * Class IntListTest
 * @package Gek\Collections\Tests\Typed
 */
class IntListTest extends TestCase
{

    public function testConstruct(): void
    {

        // array
        $arr = array(10, 15, 16, 17);
        $lst = new IntList(...$arr);
        $this->assertEquals(
            count($arr),
            $lst->count()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray()
        );

        //Enumerable
        $enmr = Enumerable::fromArray($arr);
        $lst = new IntList(...$enmr);
        $this->assertEquals(
            $enmr->count(),
            $lst->count()
        );
        $this->assertEquals(
            $enmr->toArray(),
            $lst->toArray()
        );

        // IteratorAggregate
        $arrObj = new ArrayObject($arr);
        $lst = new IntList(...$arrObj);
        $this->assertEquals(
            $arrObj->count(),
            $lst->count()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray()
        );

        // Iterator
        $itr = new ArrayIterator($arr);
        $lst = new IntList(...$itr);
        $this->assertEquals(
            $itr->count(),
            $lst->count()
        );
        $this->assertEquals(
            iterator_to_array($itr),
            $lst->toArray()
        );

        // generator
        $generator = function () {
            for ($i = 0; $i < 5; $i++) {
                yield $i;
            }
        };
        $lst = new IntList(...$generator());
        $this->assertEquals(
            5,
            $lst->count()
        );
        $this->assertEquals(
            iterator_to_array($generator()),
            $lst->toArray()
        );
    }

    public function testGetAt():void {
        $lst = new IntList(...range(0,7));
        $this->assertEquals(
            0,
            $lst->getAt(0)
        );
        $this->assertEquals(
            4,
            $lst->getAt(4)
        );
        $this->assertEquals(
            7,
            $lst->getAt(7)
        );
        $this->assertIsInt($lst->getAt(3));
        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->getAt(15);
    }

    #region IEnumerable

    public function testGetIterator(): void
    {
        $lst = new IntList();
        $this->assertInstanceOf(
            Iterator::class,
            $lst->getIterator()
        );
        $this->assertInstanceOf(
            ArrayListIterator::class,
            $lst->getIterator()
        );
    }

    public function testCount(): void
    {
        $lst = new IntList(...range(0, 10));
        $this->assertEquals(
            11,
            $lst->count()
        );
        $this->assertEquals(
            11,
            count($lst)
        );
    }

    #endregion IEnumerable

    #region ICollection

    public function testAdd(): void
    {
        $lst = new IntList();
        $this->assertEquals(
            0,
            $lst->count()
        );
        $lst->add(0);
        $this->assertEquals(
            1,
            $lst->count()
        );

        $this->assertEquals(
            0,
            $lst->getAt(0)
        );

        $lst->add(1);
        $this->assertEquals(
            2,
            $lst->count()
        );

        $this->assertEquals(
            1,
            $lst->getAt(1)
        );
        $this->assertEquals(
            [
                0 => 0,
                1 => 1
            ],
            $lst->toArray()
        );
        $this->expectException(TypeError::class);
        $lst->add('sdas');

    }

    public function testAddRange(): void
    {
        $lst = new IntList(...range(0, 5));

        $this->assertEquals(
            6,
            $lst->count()
        );

        $lst->addRange(...range(0, 5));
        $this->assertEquals(
            12,
            $lst->count()
        );

        $this->assertEquals(
            [
                0 => 0,
                1 => 1,
                2 => 2,
                3 => 3,
                4 => 4,
                5 => 5,
                6 => 0,
                7 => 1,
                8 => 2,
                9 => 3,
                10 => 4,
                11 => 5
            ],
            $lst->toArray()
        );
        $this->expectException(TypeError::class);
        $lst->addRange('dfdsf','xzdgfsgf');
    }

    public function testClear(): void
    {
        $lst = new IntList(...range(0, 10));

        $this->assertEquals(
            11,
            $lst->count()
        );
        $this->assertEquals(
            true,
            $lst->any()
        );

        $lst->clear();

        $this->assertEquals(
            0,
            $lst->count()
        );
        $this->assertEquals(
            false,
            $lst->any()
        );

    }

    public function testContains(): void
    {
        $lst = new IntList();

        $lst->add(54);
        $lst->add(78);
        $lst->add(5);

        $this->assertEquals(
            true,
            $lst->contains(54)
        );

        $this->assertEquals(
            true,
            $lst->contains(78)
        );

        $this->assertEquals(
            true,
            $lst->contains(5)
        );

        $this->assertEquals(
            false,
            $lst->contains(1)
        );
        $this->expectException(TypeError::class);
        $lst->contains('fdsfsd');
    }

    public function testRemove(): void
    {
        $lst = new IntList(...range(0, 5));
        $this->assertEquals(
            6,
            $lst->count()
        );
        $lst->remove(2);
        $this->assertEquals(
            5,
            $lst->count()
        );
        $this->assertEquals(
            3,
            $lst->getAt(2)
        );

        $lst->remove(0);
        $this->assertEquals(
            4,
            $lst->count()
        );
        $this->assertEquals(
            [
                0 => 1,
                1 => 3,
                2 => 4,
                3 => 5
            ],
            $lst->toArray()
        );
        $this->expectException(TypeError::class);
        $lst->remove('dfdsfsdf');
    }

    #endregion ICollection

    #region IList

    public function testIndexOf(): void
    {
        $lst = new IntList(...range(0, 10));
        $this->assertEquals(
            5,
            $lst->indexOf(5)
        );
        $this->assertEquals(
            -1,
            $lst->indexOf(35)
        );
        $lst->remove(4);
        $this->assertEquals(
            4,
            $lst->indexOf(5)
        );
        $lst->remove(5);
        $this->assertEquals(
            -1,
            $lst->indexOf(5)
        );
        $this->expectException(TypeError::class);
        $lst->indexOf('sgfdfsg');
    }

    public function testInsert(): void
    {
        $lst = new IntList(...range(0, 10));
        $lst->insert(5, 250);
        $this->assertEquals(
            250,
            $lst->getAt(5)
        );
        $lst->insert(0, 201);
        $this->assertEquals(
            201,
            $lst->getAt(0)
        );

        $lst->insert($lst->count(), 299);
        $this->assertEquals(
            299,
            $lst->getAt(13)
        );
        $this->assertEquals(
            [
                0 => 201,
                1 => 0,
                2 => 1,
                3 => 2,
                4 => 3,
                5 => 4,
                6 => 250,
                7 => 5,
                8 => 6,
                9 => 7,
                10 => 8,
                11 => 9,
                12 => 10,
                13 => 299
            ],
            $lst->toArray()
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->insert(52, 45645);
    }

    public function testInsertTypeError(): void
    {
        $lst = new IntList(...range(0,5));
        $this->expectException(TypeError::class);
        $lst->insert(3, 'dfgdfgdf');
    }

    public function testInsertRange(): void
    {
        $lst = new IntList(...range(0, 10));

        $lst->insertRange(5, ...[201, 202]);

        $this->assertEquals(
            201,
            $lst->getAt(5)
        );
        $this->assertEquals(
            202,
            $lst->getAt(6)
        );

        $lst->insertRange(0, ...[101, 102]);
        $this->assertEquals(
            101,
            $lst->getAt(0)
        );
        $this->assertEquals(
            102,
            $lst->getAt(1)
        );

        $lst->insertRange($lst->count(), ...[901, 902]);
        $this->assertEquals(
            901,
            $lst->getAt(15)
        );
        $this->assertEquals(
            902,
            $lst->getAt(16)
        );

        $this->assertEquals(
            [
                0 => 101,
                1 => 102,
                2 => 0,
                3 => 1,
                4 => 2,
                5 => 3,
                6 => 4,
                7 => 201,
                8 => 202,
                9 => 5,
                10 => 6,
                11 => 7,
                12 => 8,
                13 => 9,
                14 => 10,
                15 => 901,
                16 => 902
            ],
            $lst->toArray()
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->insertRange(245, ...[54654, 732434]);
    }

    public function testInsertRangeTypeError(): void
    {
        $lst = new IntList(...range(0,10));
        $this->expectException(TypeError::class);
        $lst->insertRange(2, 6, 3, 'dfgdf', 'dfdhfg');
    }

    public function testRemoveAt(): void
    {
        $lst = new IntList(...range(0, 10));

        $lst->removeAt(5);

        $this->assertEquals(
            6,
            $lst->getAt(5)
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->removeAt($lst->count());

    }

    public function testRemoveRange(): void
    {
        $lst = new IntList(...range(0, 10));
        $lst->removeRange(5, 2);
        $this->assertEquals(
            [
                0 => 0,
                1 => 1,
                2 => 2,
                3 => 3,
                4 => 4,
                5 => 7,
                6 => 8,
                7 => 9,
                8 => 10
            ],
            $lst->toArray()
        );
        $lst->removeRange(5, 4);
        $this->assertEquals(
            [
                0 => 0,
                1 => 1,
                2 => 2,
                3 => 3,
                4 => 4,
            ],
            $lst->toArray()
        );
        $lst = new IntList(...range(0, 10));
        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->removeRange(15, 1);

    }

    public function testRemoveAll(): void
    {
        $lst = new IntList(...range(0,15));



        $lst->removeAll(function ($item) {
            if($item > 0){
                return (($item % 2) != 0);
            }
            return false;
        });

        $this->assertEquals(
            [
                0 => 0,
                1 => 2,
                2 => 4,
                3 => 6,
                4 => 8,
                5 => 10,
                6 => 12,
                7 => 14,
            ],
            $lst->toArray()
        );


    }

    #endregion IList

    #region Enumerable

    public function testWhere():void {
        $lst = new IntList(...[
            10,25,18,26,21,30,20,48,5
        ]);
        $t1 = $lst->where(function ($item){
            return $item > 20;
        })->toArray();
        $this->assertEquals(
            [25,26,21,30,48],
            $t1
        );
        $t2 = $lst->where(function ($item){
            return $item < 20;
        });
        $this->assertInstanceOf(
            IEnumerable::class,
            $t2
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t2
        );
        $lst->add(8);
        $this->assertEquals(
            [10,18,5,8],
            $t2->toArray()
        );

    }

    public function testLimit():void {
        $lst = new IntList(...[
            10,25,18,26,21,30,20,48,5
        ]);
        $t1 = $lst->limit(2,2)->toArray();
        $this->assertEquals(
            [18,26],
            $t1
        );
        $t2 = $lst->limit(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t2
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t2
        );
        $lst->add(8);
        $this->assertEquals(
            [30,20,48,5,8],
            $t2->toArray()
        );

        $t3 = $lst->limit(50,10)->toArray();
        $this->assertTrue(empty($t3));
    }

    public function testSkip():void {
        $lst = new IntList(...[0,1,2,3,4,5,6,7,8,9,10]);
        $t1 = $lst->skip(0)->toArray();
        $this->assertEquals(
            [0,1,2,3,4,5,6,7,8,9,10],
            $t1
        );
        $t2 = $lst->skip(5)->toArray();
        $this->assertEquals(
            [5,6,7,8,9,10],
            $t2
        );

        $t3 = $lst->skip(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t3
        );
        $lst->add(11);
        $this->assertInstanceOf(
            Enumerable::class,
            $t3
        );
        $this->assertEquals(
            [5,6,7,8,9,10,11],
            $t3->toArray()
        );

    }

    public function testTake():void {
        $lst = new IntList(...[0,1,2,3,4,5,6,7,8,9,10]);
        $t1 = $lst->skip(0)->toArray();
        $this->assertEquals(
            [0,1,2,3,4,5,6,7,8,9,10],
            $t1
        );
        $t2 = $lst->skip(5)->toArray();
        $this->assertEquals(
            [5,6,7,8,9,10],
            $t2
        );

        $t3 = $lst->skip(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t3
        );
        $lst->add(11);
        $this->assertInstanceOf(
            Enumerable::class,
            $t3
        );
        $this->assertEquals(
            [5,6,7,8,9,10,11],
            $t3->toArray()
        );

    }

    public function testSelect(): void
    {
        $lst = new IntList(...[0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->select(function ($item) {
            return $item * 2;
        })->toArray();
        $this->assertEquals(
            [0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20],
            $t1
        );

        $lst = new ArrayList([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $lst->select(function (MockRefObj $item) {
            return $item->propA;
        })->toTypedClass(IntList::class);

        $this->assertInstanceOf(
            IntList::class,
            $t2
        );

        $this->assertEquals(
            [0, 1, 2, 3, 4, 5],
            $t2->toArray()
        );

    }

    public function testEnumerableCount():void {
        $lst = new IntList(...[0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $this->assertEquals(
            11,
            $lst->asEnumerable()->count()
        );
        $this->assertEquals(
            5,
            $lst->count(function ($item){
                return $item > 5;
            })
        );
    }

    public function testSum():void {
        $lst = new IntList(...[0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->sum();
        $this->assertEquals(
            55,
            $t1
        );

        $lst = new IntList(...[0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t2 = $lst->sum(function (int $item) {
            return $item + 1;
        });
        $this->assertEquals(
            66,
            $t2
        );
    }

    public function testAny():void {
        $lst = new IntList(...[0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        $this->assertTrue(
            $lst->any()
        );
        $this->assertTrue(
            $lst->any(function ($item){return $item > 2;})
        );
        $this->assertTrue(
            $lst->any(function ($item){return $item === 0;})
        );
        $this->assertFalse(
            $lst->any(function ($item){return $item > 15;})
        );
        $lst->clear();
        $this->assertFalse(
            $lst->any()
        );
    }

    public function testAll():void {
        $lst = new IntList(...[0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        $this->assertTrue(
            $lst->all(function ($item){return $item < 20;})
        );
        $this->assertTrue(
            $lst->all(function ($item){return is_int($item);})
        );
        $this->assertFalse(
            $lst->all(function ($item){return $item > 5;})
        );

        $lst->clear();
        $this->assertTrue(
            $lst->all(function ($item){return $item > 5;})
        );
    }

    public function testAverage():void {
        $lst = new IntList(...[0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->average();
        $this->assertEquals(
            5.0,
            $t1
        );
    }

    public function testMin():void {
        $lst = new IntList(...[0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->min();
        $this->assertEquals(
            0,
            $t1
        );
    }

    public function testMax(): void
    {
        $lst = new IntList(...[0, 1, 2, 3, 4, 5, 25, 6, 7, 8, 9, 10]);
        $t1 = $lst->max();
        $this->assertEquals(
            25,
            $t1
        );
    }

    public function testReverse(): void
    {
        $lst = new IntList(...[0, 1, 2, 3, 4, 5]);
        $t1 = $lst->reverse();
        $this->assertEquals(
            [5, 4, 3, 2, 1, 0],
            $t1->toArray()
        );
    }

    public function testAsEnumarable(): void
    {
        $lst = new IntList(...[0, 2, 3, 4, 5]);
        $t1 = $lst->where(function (int $item) {
            return $item > 3;
        });

        $t2 = $lst->where(function (int $item) {
            return $item > 3;
        })->asEnumerable();

        $lst->add(8);

        $this->assertEquals(
            [4, 5, 8],
            $t1->toArray()
        );
        $this->assertEquals(
            [4, 5],
            $t2->toArray()
        );


    }

    public function testSort(): void
    {
        $lst = new IntList(...[3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->sort();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t1->toArray()
        );
        $t2 = $lst->sort(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t2->toArray()
        );
    }

    public function testSortDesc(): void
    {
        $lst = new IntList(...[3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->sortDesc();
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t1->toArray()
        );
        $t2 = $lst->sortDesc(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t2->toArray()
        );
    }

    public function testOrderBy(): void
    {
        $lst = new IntList(...[3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->orderBy();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t1->toArray()
        );
    }

    public function testOrderByDesc(): void
    {
        $lst = new IntList(...[3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->orderByDesc();
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t1->toArray()
        );
    }

    public function testConcat(): void
    {
        $lst = new IntList(...[0, 1, 2, 3, 4, 5]);
        $t1 = $lst->concat([6, 7, 8, 9, 10]);
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
            $t1->toArray()
        );
    }

    public function testDistinct(): void
    {
        $lst = new IntList(...[0, 1, 0, 1, 1, 0]);
        $t1 = $lst->distinct();
        $this->assertEquals(
            [0, 1],
            $t1->toArray()
        );
    }

    public function testExcept(): void
    {
        $lst  = new IntList(...[0,1,2,3,4,5,6,7,8,9,10]);
        $t1 = $lst->except([10,9,5,6,7,8]);
        $this->assertEquals(
            [0,1,2,3,4],
            $t1->toArray()
        );
    }

    public function testFirstOrNUll():void {
        $lst = new IntList(...[0,1,2,3,4,5,6,7,8,9,10]);
        $this->assertEquals(
            0,
            $lst->firstOrNull()
        );
        $this->assertEquals(
            6,
            $lst->firstOrNull(function ($item){return $item > 5;})
        );
        $this->assertEquals(
            null,
            $lst->firstOrNull(function ($item){return $item > 15;})
        );
    }

    public function testAggregate(): void
    {
        $lst = new IntList(...[1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $this->assertEquals(
            3628800,
            $lst->aggregate(function (&$total, $item) {
                $total *= $item;
            })
        );
        $this->assertEquals(
            181440,
            $lst->aggregate(
                function (&$total, $item) {
                    $total *= $item;
                },
                null,
                function ($total, Enumerable $th) {
                    return ($total / $th->count()) / 2;
                })
        );
        $this->assertEquals(
            3.25,
            $lst->aggregate(
                function (&$total, $item) {
                    $total += $item;
                },
                10,
                function ($total, Enumerable $th) {
                    return ($total / $th->count()) / 2;
                })
        );
    }

    public function testUnion():void {
        $lst = new IntList(...[-3,1,2,3,3,4,4,5]);
        $t1 = $lst->union([3,8,4,8]);

        $this->assertEquals(
            [-3,1,2,3,4,5,8],
            $t1->toArray()
        );
        $t2 = $lst->union([3,8,4,8],function ($a, $b){return $a === $b;});
        $this->assertEquals(
            [-3,1,2,3,4,5,8],
            $t2->toArray()
        );
    }

    public function testIntersect():void {
        $lst = new IntList(...[1,3,4,5]);
        $t1 = $lst->intersect([3,8,4]);

        $this->assertEquals(
            [3,4],
            $t1->toArray()
        );
        $t2 = $lst->intersect([]);

        $this->assertEquals(
            [],
            $t2->toArray()
        );
    }

    public function testZip():void {
        $lst = new IntList(...[1,2,3,4,5]);
        $t1 = $lst->zip( new ArrayList(['a','b','c','d','e']), function ($f, $s){return $f . '-' . $s;});

        $this->assertEquals(
            ['1-a','2-b','3-c','4-d','5-e'],
            $t1->toArray()
        );

        $t2 = $lst->zip(new ArrayList(['a','b','c']), function ($f, $s){return $f . '-' . $s;});

        $this->assertEquals(
            ['1-a','2-b','3-c'],
            $t2->toArray()
        );

        $lst2 = new IntList(...[1,2]);
        $t3 = $lst2->zip(new ArrayList(['a','b','c','d','e']), function ($f, $s){return $f . '-' . $s;});

        $this->assertEquals(
            ['1-a','2-b'],
            $t3->toArray()
        );
    }

    public function testLastOrNull():void {
        $enmr = new IntList(...[0,1,2,3,4,5,6,7,8,9,10]);
        $t1 = $enmr->lastOrNull();
        $this->assertEquals(
            10,
            $t1
        );
        $t2 = $enmr->lastOrNull(function ($item){return $item < 5;});
        $this->assertEquals(
            4,
            $t2
        );
    }

    #endregion Enumerable

}
