<?php


namespace Gek\Collections;



use ArrayAccess;
use Gek\Collections\Traits\TypedCollectionTrait;
use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Gek\Infrastructure\Exceptions\InvalidArgumentException;
use Gek\Infrastructure\Str;

/**
 * Class TypedListWrapper
 * @package Gek\Collections
 * @method void add($item)
 * @method void addRange(...$item)
 * @method bool contains($item)
 * @method bool remove($item)
 * @method int indexOf($item)
 * @method void insert(int $index,$item)
 * @method mixed getAt(int $index)
 * @method void setAt(int $index,$item)
 * @method void insertRange(int $index,...$item)
 */
abstract class TypedListWrapper extends EnumerableWrapper implements ArrayAccess
{
    use TypedCollectionTrait;
    #region fields

    /**
     * @var IList|Enumerable
     */
    protected IList $innerList;

    #endregion fields

    #region ctor

    /**
     * TypedListWrapper constructor.
     * @param IList $list
     * @param string $itemType
     */
    protected function __construct(IList $list, string $itemType)
    {
        $this->innerList = $list;
        if(!$this->initalizeType($itemType)){
            throw new InvalidArgumentException(
                Str::format('$itemType ({0}) geçersiz tip adı.', $itemType),
                0,
                null,
                '$itemType'
            );
        }
        parent::__construct([$this,'getInnerList']);
    }

    #endregion ctor

    #region methods

    /**
     * @return IList
     */
    public function getInnerList():IList{
        return $this->innerList;
    }

    /**
     * @param int $index
     */
    public function removeAt(int $index): void{
        $this->innerList->removeAt($index);
    }

    /**
     * @param int $index
     * @param int $count
     */
    public function removeRange(int $index, int $count):void{
        $this->innerList->removeRange($index,$count);
    }

    /**
     * @param callable $fn
     * @return int
     */
    public function removeAll(callable $fn):int{
        return $this->innerList->removeAll($fn);
    }

    /**
     *
     */
    public function clear():void{
        $this->innerList->clear();
    }

    #region ArrayAccess

    /**
     * Whether a offset exists
     * @link https://php.net/manual/en/arrayaccess.offsetexists.php
     * @param mixed $offset <p>
     * An offset to check for.
     * </p>
     * @return bool true on success or false on failure.
     * </p>
     * <p>
     * The return value will be casted to boolean if non-boolean was returned.
     * @since 5.0.0
     */
    public function offsetExists($offset)
    {
        return $this->innerList->offsetExists($offset);
    }

    /**
     * Offset to retrieve
     * @link https://php.net/manual/en/arrayaccess.offsetget.php
     * @param mixed $offset <p>
     * The offset to retrieve.
     * </p>
     * @return mixed Can return all value types.
     * @since 5.0.0
     */
    public function offsetGet($offset)
    {
        return $this->innerList->offsetGet($offset);
    }

    /**
     * Offset to set
     * @link https://php.net/manual/en/arrayaccess.offsetset.php
     * @param mixed $offset <p>
     * The offset to assign the value to.
     * </p>
     * @param mixed $value <p>
     * The value to set.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetSet($offset, $value)
    {
        if (!$this->checkItemType($value)) {
            throw new InvalidArgumentException(
                Str::format('$value ("{0}"), geçerli bir "{1}" öğesi değil.', $value, $this->itemType),
                0,
                null,
                '$value'
            );
        }
        $this->innerList->offsetSet($offset,$value);

    }

    /**
     * Offset to unset
     * @link https://php.net/manual/en/arrayaccess.offsetunset.php
     * @param mixed $offset <p>
     * The offset to unset.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetUnset($offset)
    {
        $this->innerList->offsetUnset($offset);
    }

    #endregion ArrayAccess

    #endregion methods



}
