<?php


namespace Gek\Collections;

use Closure;
use Iterator;
use IteratorAggregate;
use SplFixedArray;
use Traversable;

/**
 * Class EnumerableWrapper
 * @package Gek\Collections
 */
abstract class EnumerableWrapper implements IEnumerable
{

    #region fields

    /**
     * @var Closure
     */
    private Closure $getEnumFn;

    #endregion fields

    #region ctor

    /**
     * EnumerableWrapper constructor.
     * @param callable $getEnumerableFn
     */
    protected function __construct(callable $getEnumerableFn)
    {
        $this->getEnumFn = ($getEnumerableFn instanceof Closure) ?
            $getEnumerableFn:
            Closure::fromCallable($getEnumerableFn);
    }

    #endregion ctor

    #region methods

    #region linq


    /**
     * @param callable $fn
     * @return Enumerable
     */
    public function where(callable $fn): Enumerable
    {
        return $this->getEnumerable()->where($fn);
    }

    /**
     * @param int $skip
     * @param int $take
     * @return Enumerable
     */
    public function limit(int $skip, int $take = -1): Enumerable
    {
        return $this->getEnumerable()->limit($skip, $take);
    }

    /**
     * @param int $count
     * @return Enumerable
     */
    public function skip(int $count): Enumerable
    {
        return $this->getEnumerable()->skip($count);
    }

    /**
     * @param int $count
     * @return Enumerable
     */
    public function take(int $count): Enumerable
    {
        return $this->getEnumerable()->take($count);
    }

    /**
     * @param callable $fn
     * @return Enumerable
     */
    public function select(callable $fn): Enumerable
    {
        return $this->getEnumerable()->select($fn);
    }

    /**
     * Count elements of an object
     * @link https://php.net/manual/en/countable.count.php
     * @param callable|null $fn
     * @return int The custom count as an integer.
     * </p>
     * <p>
     * The return value is cast to an integer.
     * @since 5.1.0
     */
    public function count(?callable $fn = null): int
    {
        return $this->getEnumerable()->count($fn);
    }

    /**
     * @param callable|null $fn
     * @return int|float
     */
    public function sum(?callable $fn = null)
    {
        return $this->getEnumerable()->sum($fn);
    }

    /**
     * @param callable|null $fn
     * @return bool
     */
    public function any(?callable $fn = null): bool
    {
        return $this->getEnumerable()->any($fn);
    }

    /**
     * @param callable $fn
     * @return bool
     */
    public function all(callable $fn): bool
    {
        return $this->getEnumerable()->all($fn);
    }

    /**
     * @param callable|null $fn
     * @return float
     */
    public function average(?callable $fn = null): float
    {
        return $this->getEnumerable()->average($fn);
    }

    /**
     * @param callable|null $fn
     * @return int|float|null
     */
    public function min(?callable $fn = null)
    {
        return $this->getEnumerable()->min($fn);
    }

    /**
     * @param callable|null $fn
     * @return int|float|null
     */
    public function max(?callable $fn = null)
    {
        return $this->getEnumerable()->max($fn);
    }

    /**
     * @return Enumerable
     */
    public function reverse(): Enumerable
    {
        return $this->getEnumerable()->reverse();
    }

    /**
     * @return Enumerable
     */
    public function asEnumerable(): Enumerable
    {
        return $this->getEnumerable()->asEnumerable();
    }

    /**
     * @param callable|null $compareFn
     * @return Enumerable
     */
    public function sort(?callable $compareFn = null): Enumerable
    {
        return $this->getEnumerable()->sort($compareFn);
    }

    /**
     * @param callable|null $compareFn
     * @return Enumerable
     */
    public function sortDesc(?callable $compareFn = null): Enumerable
    {
        return $this->getEnumerable()->sortDesc($compareFn);
    }

    /**
     * @param callable|null $compareFn
     * @return Enumerable
     */
    public function sortKey(?callable $compareFn = null): Enumerable
    {
        return $this->getEnumerable()->sortKey($compareFn);
    }

    /**
     * @param callable|null $compareFn
     * @return Enumerable
     */
    public function sortKeyDesc(?callable $compareFn = null): Enumerable
    {
        return $this->getEnumerable()->sortKeyDesc($compareFn);
    }

    /**
     * @param callable|null $selectorFn
     * @return Enumerable
     */
    public function orderBy(?callable $selectorFn = null): Enumerable
    {
        return $this->getEnumerable()->orderBy($selectorFn);
    }

    /**
     * @param callable|null $selectorFn
     * @return Enumerable
     */
    public function orderByDesc(?callable $selectorFn = null): Enumerable
    {
        return $this->getEnumerable()->orderByDesc($selectorFn);
    }

    /**
     * @param array|IEnumerable|IteratorAggregate|Iterator|Traversable $second
     * @return Enumerable
     */
    public function concat($second): Enumerable
    {
        return $this->getEnumerable()->concat($second);
    }

    /**
     * @param callable|null $comparerFn
     * @return Enumerable
     */
    public function distinct(?callable $comparerFn = null): Enumerable
    {
        return $this->getEnumerable()->distinct($comparerFn);
    }

    /**
     * @param array|IEnumerable|IteratorAggregate|Iterator|Traversable $second
     * @return Enumerable
     */
    public function except($second)
    {
        return $this->getEnumerable()->except($second);
    }

    /**
     * @param callable|null $fn
     * @return mixed|null
     */
    public function firstOrNull(?callable $fn = null)
    {
        return $this->getEnumerable()->firstOrNull($fn);
    }

    /**
     * @param callable $fn
     * @param mixed|null $start
     * @param callable|null $resultFn
     * @return mixed|null
     */
    public function aggregate(callable $fn, $start = null, ?callable $resultFn = null)
    {
        return $this->getEnumerable()->aggregate($fn, $start, $resultFn);
    }

    /**
     * @param array|IEnumerable|IteratorAggregate|Iterator|Traversable $second
     * @param callable|null $fn
     * @return Enumerable
     */
    public function union($second, ?callable $fn = null): Enumerable
    {
        return $this->getEnumerable()->union($second, $fn);
    }

    /**
     * @param array|IEnumerable|IteratorAggregate|Iterator|Traversable $second
     * @param callable|null $fn
     * @return Enumerable
     */
    public function intersect($second, ?callable $fn = null): Enumerable
    {
        return $this->getEnumerable()->intersect($second, $fn);
    }

    /**
     * @param array|IEnumerable|IteratorAggregate|Iterator|Traversable $second
     * @param callable $fn
     * @return Enumerable
     */
    public function zip($second, callable $fn): Enumerable
    {
        return $this->getEnumerable()->zip($second,$fn);
    }

    /**
     * @param callable|null $fn
     * @return mixed|null
     */
    public function lastOrNull(?callable $fn = null)
    {
        return $this->getEnumerable()->lastOrNull($fn);
    }

    #endregion linq

    /**
     * Retrieve an external iterator
     * @link https://php.net/manual/en/iteratoraggregate.getiterator.php
     * @return Traversable An instance of an object implementing <b>Iterator</b> or
     * <b>Traversable</b>
     * @since 5.0.0
     */
    public function getIterator()
    {
        return $this->getEnumerable()->getIterator();
    }

    /**
     * @param bool $useKeys
     * @return array
     */
    public function toArray(bool $useKeys = false): array
    {
        return $this->getEnumerable()->toArray($useKeys);
    }

    /**
     * @return SplFixedArray
     */
    public function toSplFixedArray(): SplFixedArray
    {
        return $this->getEnumerable()->toSplFixedArray();
    }

    /**
     * @return FixedArrayList
     */
    public function toFixedArrayList(): FixedArrayList
    {
        return $this->getEnumerable()->toFixedArrayList();
    }

    /**
     * @return ArrayList
     */
    public function toArrayList(): ArrayList
    {
        return $this->getEnumerable()->toArrayList();
    }

    /**
     * @return Dictionary
     */
    public function toDictionary(): Dictionary
    {
        return $this->getEnumerable()->toDictionary();
    }

    /**
     * @param $className
     * @return mixed
     */
    public function toClass($className)
    {
        return $this->getEnumerable()->toClass($className);
    }

    #endregion methods

    #region utils

    /**
     * @return Enumerable
     */
    protected function getEnumerable(): Enumerable
    {
        $fn = $this->getEnumFn;
        return $fn();
    }

    #endregion utils

}
