<?php


namespace Gek\Collections;


use Gek\Collections\Iterators\ArrayListIterator;
use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Gek\Infrastructure\Str;
use Iterator;
use IteratorAggregate;
use Traversable;
use function count;

/**
 * Class ArrayList
 * @package Gek\Collections
 */
class ArrayList extends Enumerable implements IList
{
    #region fields

    protected array $innerArray;



    #endregion fields

    #region ctor

    /**
     * FixedArrayList constructor.
     * @param Traversable|IEnumerable|null $enumerable
     */
    public function __construct($enumerable = null)
    {
        if ($enumerable === null) {
            $this->innerArray = array();
        } else {
            $this->innerArray  = $this->convertArray($enumerable);
        }
        parent::__construct(new ArrayListIterator($this));
    }

    #endregion ctor

    #region IList

    /**
     * Whether a offset exists
     * @link https://php.net/manual/en/arrayaccess.offsetexists.php
     * @param mixed $offset <p>
     * An offset to check for.
     * </p>
     * @return bool true on success or false on failure.
     * </p>
     * <p>
     * The return value will be casted to boolean if non-boolean was returned.
     * @since 5.0.0
     */
    public function offsetExists($offset)
    {
        if ($offset < 0 || $offset >= count($this->innerArray)) {
            return false;
        }
        return true;
    }

    /**
     * Offset to retrieve
     * @link https://php.net/manual/en/arrayaccess.offsetget.php
     * @param mixed $offset <p>
     * The offset to retrieve.
     * </p>
     * @return mixed Can return all value types.
     * @since 5.0.0
     */
    public function offsetGet($offset)
    {
        if ($offset < 0 || $offset >= count($this->innerArray)) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz ofset : {0}', $offset),
                0,
                null,
                '$offset'
            );
        }
        return $this->innerArray[$offset];
    }

    /**
     * Offset to set
     * @link https://php.net/manual/en/arrayaccess.offsetset.php
     * @param mixed $offset <p>
     * The offset to assign the value to.
     * </p>
     * @param mixed $value <p>
     * The value to set.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetSet($offset, $value)
    {

        if ($offset < 0 || $offset >= count($this->innerArray)) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz ofset : {0}', $offset),
                0,
                null,
                '$offset'
            );
        }
        $this->innerArray[$offset] = $value;
    }

    /**
     * Offset to unset
     * @link https://php.net/manual/en/arrayaccess.offsetunset.php
     * @param mixed $offset <p>
     * The offset to unset.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetUnset($offset)
    {
        if ($offset < 0 || $offset >= count($this->innerArray)) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz ofset : {0}', $offset),
                0,
                null,
                '$offset'
            );
        }
        unset($this->innerArray[$offset]);
        $this->innerArray = array_values($this->innerArray);
    }

    /**
     * @param mixed $item
     */
    public function add($item): void
    {
        $this->innerArray[] = $item;
    }

    /**
     * @param array|Enumerable|\IteratorAggregate|\Iterator|\Traversable $items
     */
    public function addRange($items): void
    {
        $items = $this->convertArray($items);
        $tmp = array_merge($this->innerArray,$items);
        $this->innerArray = $tmp;
    }

    /**
     *
     */
    public function clear(): void
    {
        $this->innerArray = array();
    }

    /**
     * {@inheritDoc}
     */
    public function contains($item): bool
    {
        for ($i = 0; $i < count($this->innerArray); $i++){
            $fullEqual = !is_object($item) || !is_object($this->innerArray[$i]);
            $res = $fullEqual ?
                ($this->innerArray[$i] === $item):
                ($this->innerArray[$i] == $item);
            if($res){
                return true;
            }
        }
        return false;
    }

    /**
     * @param mixed $item
     * @return bool
     */
    public function remove($item): bool
    {
        $index = $this->indexOf($item);
        if ($index == -1) {
            return false;
        }

        $this->removeAt($index);
        return true;
    }

    /**
     * @param mixed $item
     * @return int
     */
    public function indexOf($item): int
    {
        for ($i = 0; $i < count($this->innerArray); $i++) {

            if ($this->innerArray[$i] == $item) {
                return $i;
            }
        }
        return -1;
    }

    /**
     * @param int $index
     * @param mixed $item
     */
    public function insert(int $index, $item): void
    {
        if ($index < 0 || $index > count($this->innerArray)) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz index : {0}', $index),
                0,
                null,
                '$index'
            );
        }
        if($index == count($this->innerArray)){
            $this->innerArray[$index] = $item;
        }elseif ($index == 0){
            $this->innerArray = array_merge([$item],$this->innerArray);
        }else{
            $tmp1 = array_slice($this->innerArray,0,$index + 1);
            $tmp2 = array_slice($this->innerArray,$index);
            $tmp1[$index] =  $item;
            $this->innerArray = array_merge($tmp1,$tmp2);
        }
    }

    /**
     * @param int $index
     * @param array|Enumerable|\IteratorAggregate|\Iterator|\Traversable $items
     */
    public function insertRange(int $index, $items): void
    {
        if ($index < 0 || $index > count($this->innerArray)) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz index : {0}', $index),
                0,
                null,
                '$index'
            );
        }
        $items = $this->convertArray($items);

        if($index == $this->count()){
            $this->innerArray = array_merge($this->innerArray, $items);
        }elseif ($index == 0){
            $this->innerArray = array_merge($items,$this->innerArray);
        }else{
            $tmp1 = array_slice($this->innerArray,0,$index);
            $tmp2 = array_slice($this->innerArray,$index);
            $tmp1 =  array_merge($tmp1,$items);
            $this->innerArray = array_merge($tmp1,$tmp2);
        }
    }

    /**
     * @param int $index
     */
    public function removeAt(int $index): void
    {
        if ($index < 0 || $index >= count($this->innerArray)) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz index : {0}', $index),
                0,
                null,
                '$index'
            );
        }
        array_splice($this->innerArray,$index,1);
    }

    /**
     * @param int $index
     * @param int $count
     */
    public function removeRange(int $index, int $count): void
    {
        if ($index < 0 || $index >= count($this->innerArray)) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz index : {0}', $index),
                0,
                null,
                '$index'
            );
        }
        $lastIndex = ($index + $count) - 1;
        if ($lastIndex < 0 || $lastIndex >= count($this->innerArray)) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz count : {0}', $count),
                0,
                null,
                '$count'
            );
        }
        array_splice($this->innerArray,$index,$count);
    }

    /**
     * @param callable $fn
     * @return int
     */
    public function removeAll(callable $fn): int
    {
        $rmvItems = $this->where($fn)->toArray();
        $removedCount = 0;
        foreach ($rmvItems as $key => $val){
            $this->remove($val);
            $removedCount++;
        }
        unset($rmvItems);
        return $removedCount;
    }

    /**
     * @param callable|null $fn
     * @return int
     */
    public function count(?callable $fn = null): int
    {
        if ($fn === null) {
            return count($this->innerArray);
        }
        return parent::count($fn);
    }

    #endregion IList

    #region Enumerable Overrides
    /**
     * @param callable|null $fn
     * @return mixed|null
     */
    public function firstOrNull(?callable $fn = null)
    {
        if(count($this->innerArray) == 0){
            return null;
        }
        if($fn === null){
            return $this[0];
        }
        return parent::firstOrNull($fn);
    }

    /**
     * @param callable|null $fn
     * @return mixed|null
     */
    public function lastOrNull(?callable $fn = null)
    {
        if(count($this->innerArray) == 0){
            return null;
        }
        if($fn === null){
            return $this[count($this->innerArray) - 1];
        }
        for ($i = count($this->innerArray) - 1; $i >= 0; $i--){
            if($fn($this[$i],$i,$this->iterator) === true){
                return $this[$i];
            }
        }
        return null;
    }

    #endregion Enumerable Overrides



    #region utils

    /**
     * @param array|IEnumerable|Enumerable|IteratorAggregate|Iterator|Traversable|object $items
     * @return array
     */
    protected function convertArray($items):array {
        if(!is_array($items)){
            if($items instanceof Enumerable){
                $items = $items->toArray();
            }elseif ($items instanceof IteratorAggregate){
                $items = iterator_to_array($items->getIterator());
            }elseif ($items instanceof Iterator){
                $items = iterator_to_array($items);
            }else{
                $tmpArr = array();
                foreach ($items as $itm){
                    $tmpArr[] = $itm;
                }
                $items = $tmpArr;
            }
        }
        return array_values($items);
    }

    #endregion utils

}
