<?php


namespace Gek\Collections\Iterators;


use Closure;
use Iterator;

/**
 * Class DistinctIterator
 * @package Gek\Collections\Iterators
 */
class DistinctIterator implements Iterator
{
    #region fields

    protected Iterator $iterator;

    protected array $tempArray = array();

    protected ?Closure $comparerFn = null;

    #endregion fields

    #region ctor

    /**
     * DistinctIterator constructor.
     * @param Iterator $iterator
     * @param callable|null $comparerFn
     */
    public function __construct(Iterator $iterator, ?callable $comparerFn = null)
    {
        $this->iterator = $iterator;
        if($comparerFn !== null){
            $this->comparerFn = ($comparerFn instanceof Closure) ? $comparerFn : Closure::fromCallable($comparerFn);
        }
    }

    #endregion ctor

    #region Iterator

    /**
     * Return the current element
     * @link https://php.net/manual/en/iterator.current.php
     * @return mixed Can return any type.
     * @since 5.0.0
     */
    public function current()
    {
        $item = $this->iterator->current();
        $this->tempArray[] = $item;
        return $item;
    }

    /**
     * Move forward to next element
     * @link https://php.net/manual/en/iterator.next.php
     * @return void Any returned value is ignored.
     * @since 5.0.0
     */
    public function next()
    {
        $this->iterator->next();
        if($this->comparerFn === null){
            while($this->iterator->valid() && in_array($this->iterator->current(),$this->tempArray)){
                $this->iterator->next();
            }
        }else{
            $found = true;
            while($this->iterator->valid() && $found){
                $found = false;
                foreach ($this->tempArray as $itm){
                    $found = $this->comparerFn->call($this,$this->iterator->current(), $itm);
                    if($found == true){
                        break;
                    }
                }
                if($found){
                    $this->iterator->next();
                }
            }
        }

    }

    /**
     * Return the key of the current element
     * @link https://php.net/manual/en/iterator.key.php
     * @return mixed scalar on success, or null on failure.
     * @since 5.0.0
     */
    public function key()
    {
        return $this->iterator->key();
    }

    /**
     * Checks if current position is valid
     * @link https://php.net/manual/en/iterator.valid.php
     * @return bool The return value will be casted to boolean and then evaluated.
     * Returns true on success or false on failure.
     * @since 5.0.0
     */
    public function valid()
    {
        return $this->iterator->valid();
    }

    /**
     * Rewind the Iterator to the first element
     * @link https://php.net/manual/en/iterator.rewind.php
     * @return void Any returned value is ignored.
     * @since 5.0.0
     */
    public function rewind()
    {
        $this->tempArray = array();
        $this->iterator->rewind();
    }

    #endregion Iterator

}
