<?php


namespace Gek\Collections;


use AppendIterator;
use ArrayIterator;
use CallbackFilterIterator;
use Closure;
use EmptyIterator;
use Gek\Collections\Iterators\DistinctIterator;
use Gek\Collections\Iterators\ExceptIterator;
use Gek\Collections\Iterators\IntersectIterator;
use Gek\Collections\Iterators\ReverseIterator;
use Gek\Collections\Iterators\SelectIterator;
use Gek\Collections\Iterators\SortIterator;
use Gek\Collections\Iterators\UnionIterator;
use Gek\Collections\Iterators\ZipIterator;
use Gek\Infrastructure\IComparable;
use Iterator;
use IteratorAggregate;
use LimitIterator;
use SplFixedArray;
use Traversable;

/**
 * Class Enumerable
 * @package Gek\Collections
 */
class Enumerable implements IEnumerable
{

    #region fields

    protected Iterator $iterator;

    #endregion fields

    #region ctor
    /**
     * Enumerable constructor.
     * @param Iterator $iterator
     */
    protected function __construct(Iterator $iterator)
    {
        $this->iterator = $iterator;
    }

    #endregion ctor

    #region IteratorAggregate

    /**
     * Retrieve an external iterator
     * @link https://php.net/manual/en/iteratoraggregate.getiterator.php
     * @return Traversable An instance of an object implementing <b>Iterator</b> or
     * <b>Traversable</b>
     * @since 5.0.0
     */
    public function getIterator(): Iterator
    {
        return $this->iterator;
    }

    #endregion IteratorAggregate

    #region linq


    /**
     * @param callable $fn
     * @return Enumerable
     */
    public function where(callable $fn): Enumerable
    {
        return self::createInstance(new CallbackFilterIterator($this->iterator, $fn));
    }

    /**
     * @param int $skip
     * @param int $take
     * @return Enumerable
     */
    public function limit(int $skip, int $take = -1): Enumerable
    {

        //$skip = $skip > 0 ? ($skip - 1) : $skip;
        return self::createInstance(new LimitIterator($this->iterator, $skip, $take));
    }

    /**
     * @param int $count
     * @return Enumerable
     */
    public function skip(int $count): Enumerable
    {
        return $this->limit($count);
    }

    /**
     * @param int $count
     * @return Enumerable
     */
    public function take(int $count): Enumerable
    {
        return $this->limit(0, $count);
    }

    /**
     * @param callable $fn
     * @return Enumerable
     */
    public function select(callable $fn): Enumerable
    {
        return self::createInstance(new SelectIterator($this->iterator, $fn));
    }

    /**
     * Count elements of an object
     * @link https://php.net/manual/en/countable.count.php
     * @param callable|null $fn
     * @return int The custom count as an integer.
     * </p>
     * <p>
     * The return value is cast to an integer.
     * @since 5.1.0
     */
    public function count(?callable $fn = null): int
    {
        $count = 0;
        if ($fn === null) {
            if ($this->iterator instanceof ArrayIterator) {
                return $this->iterator->count();
            }

            foreach ($this as $item) {

                $count++;
            }

        } else {
            $count = 0;
            foreach ($this as $key => $item) {
                if ($fn($item, $key, $this->iterator)) {
                    $count++;
                }
            }
        }
        return $count;
    }

    /**
     * @param callable|null $fn
     * @return int|float
     */
    public function sum(?callable $fn = null)
    {
        $sum = 0.0;
        if ($fn !== null) {
            foreach ($this as $key => $item) {
                $res = $fn($item, $key, $this->iterator);
                if ($res !== null) {
                    $sum += $res;
                }
            }
        } else {
            foreach ($this as $item) {

                if ($item !== null) {
                    $sum += $item;
                }
            }
        }
        return $sum;
    }

    /**
     * @param callable|null $fn
     * @return bool
     */
    public function any(?callable $fn = null): bool
    {
        if ($fn !== null) {
            foreach ($this as $key => $item) {
                if ($fn($item, $key, $this->iterator) == true) {
                    return true;
                }
            }
        } else {
            foreach ($this as $key => $item) {
                return true;
            }
        }
        return false;
    }

    /**
     * @param callable $fn
     * @return bool
     */
    public function all(callable $fn): bool
    {
        foreach ($this as $key => $item) {
            if ($fn($item, $key, $this->iterator) == false) {
                return false;
            }
        }
        return true;
    }

    /**
     * @param callable|null $fn
     * @return float
     */
    public function average(?callable $fn = null): float
    {
        $count = 0;
        $sum = 0.0;

        if ($fn !== null) {
            foreach ($this as $key => $item) {
                $count++;
                $res = $fn($item, $key, $this->iterator);
                if ($res !== null) {
                    $sum += $res;
                }
            }
        } else {
            foreach ($this as $key => $item) {
                $count++;
                if ($item !== null) {
                    $sum += $item;
                }
            }
        }
        return (float)($sum / $count);
    }

    /**
     * @param callable|null $fn
     * @return int|float|null
     */
    public function min(?callable $fn = null)
    {
        $min = null;
        if ($fn !== null) {
            foreach ($this as $key => $item) {
                $res = $fn($item, $key, $this->iterator);
                if ($min === null) {
                    $min = $res;
                }
                if ($min > $res) {
                    $min = $res;
                }
            }
        } else {
            foreach ($this as $item) {

                if ($min === null) {
                    $min = $item;
                }
                if ($min > $item) {
                    $min = $item;
                }
            }
        }
        return $min;
    }

    /**
     * @param callable|null $fn
     * @return int|float|null
     */
    public function max(?callable $fn = null)
    {
        $max = null;
        if ($fn !== null) {
            foreach ($this as $key => $item) {
                $res = $fn($item, $key, $this->iterator);
                if ($max === null) {
                    $max = $res;
                }
                if ($max < $res) {
                    $max = $res;
                }
            }
        } else {
            foreach ($this as $item) {

                if ($max === null) {
                    $max = $item;
                }
                if ($max < $item) {
                    $max = $item;
                }
            }
        }
        return $max;
    }

    /**
     * @return Enumerable
     */
    public function reverse(): Enumerable
    {
        return self::createInstance(new ReverseIterator($this->iterator));
    }

    /**
     * @return Enumerable
     */
    public function asEnumerable(): Enumerable
    {
        $tempArr = iterator_to_array($this->iterator);
        return self::createInstance(new ArrayIterator($tempArr));
    }

    /**
     * @param callable|null $compareFn
     * @return Enumerable
     */
    public function sort(?callable $compareFn = null): Enumerable
    {
        if ($compareFn === null) {
            $compareFn = $this->createCompareFn();
        }
        return self::createInstance(new SortIterator($this->iterator, $compareFn));
    }

    /**
     * @param callable|null $compareFn
     * @return Enumerable
     */
    public function sortDesc(?callable $compareFn = null): Enumerable
    {
        if ($compareFn === null) {
            $compareFn = $this->createCompareFn();
        }
        return self::createInstance(new SortIterator($this->iterator, $compareFn, true));
    }

    /**
     * @param callable|null $compareFn
     * @return Enumerable
     */
    public function sortKey(?callable $compareFn = null): Enumerable
    {
        if ($compareFn === null) {
            $compareFn = $this->createCompareFn();
        }
        return self::createInstance(new SortIterator($this->iterator, $compareFn, false, true));
    }

    /**
     * @param callable|null $compareFn
     * @return Enumerable
     */
    public function sortKeyDesc(?callable $compareFn = null): Enumerable
    {
        if ($compareFn === null) {
            $compareFn = $this->createCompareFn();
        }
        return self::createInstance(new SortIterator($this->iterator, $compareFn, true, true));
    }

    /**
     * @param callable|null $selectorFn
     * @return Enumerable
     */
    public function orderBy(?callable $selectorFn = null): Enumerable
    {
        $compareFn = $this->createCompareFn($selectorFn);
        return $this->sort($compareFn);
    }

    /**
     * @param callable|null $selectorFn
     * @return Enumerable
     */
    public function orderByDesc(?callable $selectorFn = null): Enumerable
    {
        $compareFn = $this->createCompareFn($selectorFn);
        return $this->sortDesc($compareFn);
    }

    /**
     * @param array|IEnumerable|IteratorAggregate|Iterator|Traversable $second
     * @return Enumerable
     */
    public function concat($second): Enumerable
    {
        $iterator = new AppendIterator();
        $iterator->append($this->iterator);
        if ($second instanceof Iterator) {
            $iterator->append($second);
        } elseif ($second instanceof IteratorAggregate) {
            $itr = $second->getIterator();
            $iterator->append($itr);
        } elseif (is_array($second)) {
            $iterator->append(new ArrayIterator($second));
        } else {
            $temArr = array();
            foreach ($second as $key => $item) {
                $temArr[$key] = $item;
            }
            $iterator->append(new ArrayIterator($second));
        }
        return self::createInstance($iterator);
    }

    /**
     * @param callable|null $comparerFn
     * @return Enumerable
     */
    public function distinct(?callable $comparerFn = null): Enumerable
    {
        return self::createInstance(new DistinctIterator($this->iterator, $comparerFn));
    }

    /**
     * @param array|IEnumerable|IteratorAggregate|Iterator|Traversable $second
     * @return $this
     */
    public function except($second)
    {
        $secondIterator = $this->convertToIterator($second);
        return self::createInstance(new ExceptIterator($this->iterator, $secondIterator));
    }

    /**
     * @param callable|null $fn
     * @return mixed|null
     */
    public function firstOrNull(?callable $fn = null)
    {
        if ($fn !== null) {
            foreach ($this as $key => $item) {
                if ($fn($item, $key, $this->iterator)) {
                    return $item;
                }
            }
        } else {
            foreach ($this as $item) {
                return $item;
            }
        }
        return null;
    }

    /**
     * @param callable $fn
     * @param mixed|null $start
     * @param callable|null $resultFn
     * @return mixed|null
     */
    public function aggregate(callable $fn, $start = null, ?callable $resultFn = null)
    {
        $total = $start;
        $isSetTotal = $total !== null ? true : false;
        foreach ($this as $itm) {
            if (!$isSetTotal) {
                $total = $itm;
                $isSetTotal = true;
            } else {
                $fn($total, $itm);
            }
        }
        if ($resultFn !== null) {
            return $resultFn($total, $this);
        }
        return $total;
    }

    /**
     * @param array|IEnumerable|IteratorAggregate|Iterator|Traversable $second
     * @param callable|null $fn
     * @return Enumerable
     */
    public function union($second, ?callable $fn = null):Enumerable{
        $secondIterator = $this->convertToIterator($second);
        return self::createInstance(new UnionIterator($this->iterator,$secondIterator,$fn));
    }

    /**
     * @param array|IEnumerable|IteratorAggregate|Iterator|Traversable $second
     * @param callable|null $fn
     * @return Enumerable
     */
    public function intersect($second, ?callable $fn = null):Enumerable{
        $secondIterator = $this->convertToIterator($second);
        return self::createInstance(new IntersectIterator($this->iterator,$secondIterator,$fn));
    }

    /**
     * @param array|IEnumerable|IteratorAggregate|Iterator|Traversable $second
     * @param callable $fn
     * @return Enumerable
     */
    public function zip($second, callable $fn):Enumerable{
        $secondIterator = $this->convertToIterator($second);
        return self::createInstance(new ZipIterator($this->iterator,$secondIterator,$fn));
    }

    /**
     * @param callable|null $fn
     * @return mixed|null
     */
    public function lastOrNull(?callable $fn = null){
        return $this->reverse()->firstOrNull($fn);
    }

    // LastOrNull

    // Union

    //

    #endregion linq

    /**
     * @param bool $useKeys
     * @return array
     */
    public function toArray(bool $useKeys = false): array
    {
        return iterator_to_array($this->iterator, $useKeys);
    }

    /**
     * @return SplFixedArray
     */
    public function toSplFixedArray(): SplFixedArray
    {
        return SplFixedArray::fromArray($this->toArray(), false);
    }

    /**
     * @return FixedArrayList
     */
    public function toFixedArrayList():FixedArrayList{
        return new FixedArrayList($this);
    }

    /**
     * @return ArrayList
     */
    public function toArrayList():ArrayList{
        return new ArrayList($this);
    }


    /**
     * @return Dictionary
     */
    public function toDictionary():Dictionary{
        return new Dictionary($this);
    }

    #region utils

    /**
     * @param array|IEnumerable|IteratorAggregate|Iterator|Traversable|object $mixed
     * @return Iterator
     */
    protected function convertToIterator($mixed):Iterator{

        if ($mixed instanceof Iterator) {
           return $mixed;
        } elseif ($mixed instanceof IteratorAggregate) {
            return $mixed->getIterator();
        } elseif (is_array($mixed)) {
           return new ArrayIterator($mixed);
        } else {
            $temArr = array();
            foreach ($mixed as $key => $item) {
                $temArr[$key] = $item;
            }
            return new ArrayIterator($temArr);
        }
    }

    /**
     * @param Iterator $iterator
     */
    protected function injectIterator(Iterator $iterator): void
    {
        $this->iterator = $iterator;
    }

    /**
     * @param callable|null $selectFn
     * @return Closure
     */
    protected function createCompareFn(?callable $selectFn = null): Closure
    {
        if ($selectFn != null) {
            return function ($a, $b) use ($selectFn) {
                $aVal = $selectFn($a);
                $bVal = $selectFn($b);
                if (($aVal instanceof IComparable)) {
                    return $aVal->compareTo($bVal);
                }
                if ($aVal == $bVal) {
                    return 0;
                }
                return ($aVal < $bVal) ? -1 : 1;
            };
        }
        return function ($a, $b) {
            if (($a instanceof IComparable)) {
                return $a->compareTo($b);
            }
            if ($a == $b) {
                return 0;
            }
            return ($a < $b) ? -1 : 1;
        };
    }

    /**
     * @param Iterator $iterator
     * @return static
     */
    protected static function createInstance(Iterator $iterator): self
    {
        static $enumerableInstance = null;
        if ($enumerableInstance === null) {
            $enumerableInstance = new Enumerable(new EmptyIterator());
        }
        $newInstance = clone $enumerableInstance;
        $newInstance->injectIterator($iterator);
        return $newInstance;
    }

    #endregion utils

    /**
     * @param array $arr
     * @return Enumerable
     */
    public static function fromArray(array $arr): Enumerable
    {
        return self::createInstance(new ArrayIterator($arr));
    }

    /**
     * @param SplFixedArray $fixedArray
     * @return static
     */
    public static function fromSplFixedArray(SplFixedArray $fixedArray)
    {
        return self::createInstance($fixedArray);
    }


}
