<?php


namespace Gek\Collections\Iterators;


use Gek\Collections\Dictionary;
use Iterator;

/**
 * Class DictionaryIterator
 * @package Gek\Collections\Iterators
 */
class DictionaryIterator implements Iterator
{

    #region fields

    protected Dictionary $dictionary;

    protected ?array $keys = null;

    protected int $position = 0;

    #endregion fields

    #region ctor
    /**
     * DictionaryIterator constructor.
     * @param Dictionary $dictionary
     */
    public function __construct(Dictionary $dictionary)
    {
        $this->dictionary = $dictionary;
    }

    #endregion ctor

    #region Iterator

    /**
     * Return the current element
     * @link https://php.net/manual/en/iterator.current.php
     * @return mixed Can return any type.
     * @since 5.0.0
     */
    public function current()
    {
        $this->check();
        return $this->dictionary[$this->keys[$this->position]];
    }

    /**
     * Move forward to next element
     * @link https://php.net/manual/en/iterator.next.php
     * @return void Any returned value is ignored.
     * @since 5.0.0
     */
    public function next()
    {
        $this->check();
        $this->position++;
    }

    /**
     * Return the key of the current element
     * @link https://php.net/manual/en/iterator.key.php
     * @return mixed scalar on success, or null on failure.
     * @since 5.0.0
     */
    public function key()
    {
        $this->check();
        return $this->keys[$this->position];
    }

    /**
     * Checks if current position is valid
     * @link https://php.net/manual/en/iterator.valid.php
     * @return bool The return value will be casted to boolean and then evaluated.
     * Returns true on success or false on failure.
     * @since 5.0.0
     */
    public function valid()
    {
        $this->check();
        return $this->position < count($this->keys);
    }

    /**
     * Rewind the Iterator to the first element
     * @link https://php.net/manual/en/iterator.rewind.php
     * @return void Any returned value is ignored.
     * @since 5.0.0
     */
    public function rewind()
    {
        $this->keys = null;
        $this->position = 0;
    }

    #endregion Iterator

    #region utils

    public function check(){
        if($this->keys !== null){
            return;
        }
        $this->keys = $this->dictionary->getKeys()->toArray();
    }

    #endregion utils

}
