<?php


namespace Gek\Collections;


use Gek\Collections\Iterators\DictionaryIterator;
use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Gek\Infrastructure\Str;
use Iterator;
use IteratorAggregate;
use Traversable;

/**
 * Class Dictionary
 * @package Gek\Collections
 */
class Dictionary extends Enumerable implements IDictionary
{
    #region fields

    protected array $innerArray;

    #endregion fields

    #region ctor

    /**
     * FixedArrayList constructor.
     * @param array|Traversable|IEnumerable|object|null $enumerable
     */
    public function __construct($enumerable = null)
    {
        if ($enumerable === null) {
            $this->innerArray = array();
        } else {
            $this->innerArray  = $this->convertArray($enumerable);
        }
        parent::__construct(new DictionaryIterator($this));

    }

    #endregion ctor

    #region IDictionary

    /**
     * Whether a offset exists
     * @link https://php.net/manual/en/arrayaccess.offsetexists.php
     * @param mixed $offset <p>
     * An offset to check for.
     * </p>
     * @return bool true on success or false on failure.
     * </p>
     * <p>
     * The return value will be casted to boolean if non-boolean was returned.
     * @since 5.0.0
     */
    public function offsetExists($offset)
    {
        return isset($this->innerArray[$offset]);
    }

    /**
     * Offset to retrieve
     * @link https://php.net/manual/en/arrayaccess.offsetget.php
     * @param mixed $offset <p>
     * The offset to retrieve.
     * </p>
     * @return mixed Can return all value types.
     * @since 5.0.0
     */
    public function offsetGet($offset)
    {
        if (!isset($this->innerArray[$offset])) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz ofset : {0}', $offset),
                0,
                null,
                '$offset'
            );
        }
        return $this->innerArray[$offset];
    }

    /**
     * Offset to set
     * @link https://php.net/manual/en/arrayaccess.offsetset.php
     * @param mixed $offset <p>
     * The offset to assign the value to.
     * </p>
     * @param mixed $value <p>
     * The value to set.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetSet($offset, $value)
    {
        if (!isset($this->innerArray[$offset])) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz ofset : {0}', $offset),
                0,
                null,
                '$offset'
            );
        }
        $this->innerArray[$offset] = $value;
    }

    /**
     * Offset to unset
     * @link https://php.net/manual/en/arrayaccess.offsetunset.php
     * @param mixed $offset <p>
     * The offset to unset.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetUnset($offset)
    {
        if (!isset($this->innerArray[$offset])) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz ofset : {0}', $offset),
                0,
                null,
                '$offset'
            );
        }
        unset($this->innerArray[$offset]);
    }

    /**
     * @param mixed $item
     */
    public function add($item): void
    {
        $this->innerArray[] = $item;
    }

    /**
     * @param array|Enumerable|IteratorAggregate|\Iterator|Traversable $items
     */
    public function addRange($items): void
    {
        $items = $this->convertArray($items);
        foreach ($items as $key => $item){
            if(isset($this->innerArray[$key])){
                throw new ArgumentOutOfRangeException(
                    Str::format('Geçersiz ofset : {0}', $key),
                    0,
                    null,
                    '$items'
                );
            }
            $this->innerArray[$key] =  $item;
        }
    }

    /**
     *
     */
    public function clear(): void
    {
        $this->innerArray = array();
    }

    /**
     * @param mixed $item
     * @return bool
     */
    public function contains($item): bool
    {
        foreach ($this->innerArray as $itm){
            $fullEqual = !is_object($item) || !is_object($itm);
            $res = $fullEqual ?
                ($itm === $item):
                ($itm == $item);
            if($res){
                return true;
            }
        }
        return false;
    }

    /**
     * @param mixed $item
     * @return bool
     */
    public function remove($item): bool
    {
        foreach ($this->innerArray  as $key => $itm){
            $fullEqual = !is_object($item) || !is_object($this->innerArray[$itm]);
            $res = $fullEqual ?
                ($itm === $item):
                ($itm == $item);
            if($res){
                unset($this->innerArray[$key]);
                return true;
            }
        }
        return false;
    }

    /**
     * @param mixed $key
     * @param mixed $value
     */
    public function addKeyValue($key, $value): void
    {
        if (isset($this->innerArray[$key])) {
            throw new ArgumentOutOfRangeException(
                Str::format('Key zaten eklenmiş : {0}', $key),
                0,
                null,
                '$key'
            );
        }
        $this->innerArray[$key] = $value;
    }

    /**
     * @param $key
     * @param $value
     * @return bool
     */
    public function tryAddKeyValue($key,$value):bool {
        if (isset($this->innerArray[$key])) {
            return false;
        }
        $this->innerArray[$key] = $value;
        return true;
    }

    /**
     * @param $key
     * @return bool
     */
    public function containsKey($key): bool
    {
       return isset($this->innerArray[$key]);
    }

    /**
     * @param mixed $key
     * @return bool
     */
    public function removeByKey($key): bool
    {
        if(isset($this->innerArray[$key])){
            unset($this->innerArray[$key]);
            return true;
        }
        return false;
    }


    /**
     * @return Enumerable
     */
    public function getKeys(): Enumerable
    {
        return Enumerable::fromArray(array_keys($this->innerArray));
    }

    /**
     * @return Enumerable
     */
    public function getValues(): Enumerable
    {
        return Enumerable::fromArray(array_values($this->innerArray));
    }

    /**
     * @param callable|null $fn
     * @return int
     */
    public function count(?callable $fn = null): int
    {
        if($fn === null){
            return count($this->innerArray);
        }

        return parent::count($fn);
    }

    #endregion IDictionary

    #region Utils

    /**
     * @param array|IEnumerable|Enumerable|IteratorAggregate|Iterator|Traversable|object $items
     * @return array
     */
    protected function convertArray($items):array {
        if(!is_array($items)){
            if($items instanceof Enumerable){
                $items = $items->toArray(true);
            }elseif ($items instanceof IteratorAggregate){
                $items = iterator_to_array($items->getIterator(),true);
            }elseif ($items instanceof Iterator){
                $items = iterator_to_array($items, true);
            }else{
                $tmpArr = array();
                foreach ($items as $key => $itm){
                    $tmpArr[$key] = $itm;
                }
                $items = $tmpArr;
            }
        }
        return $items;
    }

    #endregion Utils
}
