<?php


namespace Gek\Collections\Tests;


use Gek\Collections\ArrayList;
use Gek\Collections\Dictionary;
use Gek\Collections\Enumerable;
use Gek\Collections\FixedArrayList;
use Gek\Collections\IEnumerable;
use Gek\Collections\Tests\Fixtures\MockRefObj;
use Gek\Infrastructure\Math\Decimal;
use OutOfBoundsException;
use PHPUnit\Framework\TestCase;
use SplFixedArray;
use function Webmozart\Assert\Tests\StaticAnalysis\null;

/**
 * Class EnumerableTest
 * @package Gek\Collections\Tests
 */
class EnumerableTest extends TestCase
{
    public function testWhere(): void
    {
        $enmb = Enumerable::fromArray([10, 25, 18, 26, 21, 30, 20, 48, 5]);
        $t1 = $enmb->where(function ($item) {
            return $item > 20;
        })->toArray();
        $this->assertEquals(
            [25, 26, 21, 30, 48],
            $t1
        );
        $t2 = $enmb->where(function ($item) {
            return $item < 20;
        });
        $this->assertInstanceOf(
            IEnumerable::class,
            $t2
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t2
        );
        $this->assertEquals(
            [10, 18, 5],
            $t2->toArray()
        );
    }

    public function testLimit(): void
    {
        $enmb = Enumerable::fromArray([
            10, 25, 18, 26, 21, 30, 20, 48, 5
        ]);
        $t1 = $enmb->limit(2, 2)->toArray();
        $this->assertEquals(
            [18, 26],
            $t1
        );
        $t2 = $enmb->limit(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t2
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t2
        );
        $this->assertEquals(
            [30, 20, 48, 5],
            $t2->toArray()
        );
        $enmb2 = Enumerable::fromSplFixedArray(SplFixedArray::fromArray([10, 25, 18, 26, 21, 30, 20, 48, 5]));
        $t3 = $enmb2->limit(50, 10)->toArray();
        $this->assertTrue(empty($t3));

        $this->expectException(OutOfBoundsException::class);
        $enmb->limit(50, 10)->toArray();

    }

    public function testSkip(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmb->skip(0)->toArray();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
            $t1
        );
        $t2 = $enmb->skip(5)->toArray();
        $this->assertEquals(
            [5, 6, 7, 8, 9, 10],
            $t2
        );

        $t3 = $enmb->skip(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t3
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t3
        );
        $this->assertEquals(
            [5, 6, 7, 8, 9, 10],
            $t3->toArray()
        );
    }

    public function testTake(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmb->skip(0)->toArray();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
            $t1
        );
        $t2 = $enmb->skip(5)->toArray();
        $this->assertEquals(
            [5, 6, 7, 8, 9, 10],
            $t2
        );

        $t3 = $enmb->skip(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t3
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t3
        );
        $this->assertEquals(
            [5, 6, 7, 8, 9, 10],
            $t3->toArray()
        );

    }

    public function testSelect(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmb->select(function ($item) {
            return $item * 2;
        })->toArray();
        $this->assertEquals(
            [0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20],
            $t1
        );

        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $enmb2->select(function (MockRefObj $item) {
            return $item->propA;
        })->toArray();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5],
            $t2
        );
        $t3 = $enmb2->select(function (MockRefObj $item) {
            return $item->propB;
        })->toArray();
        $this->assertEquals(
            [5, 5, 5, 5, 5, 5],
            $t3
        );
    }

    public function testCount(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $this->assertEquals(
            11,
            $enmb->count()
        );
        $this->assertEquals(
            5,
            $enmb->count(function ($item) {
                return $item > 5;
            })
        );

    }

    public function testSum(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmb->sum();
        $this->assertEquals(
            55,
            $t1
        );

        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $enmb2->sum(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            15,
            $t2
        );
    }

    public function testSumDecimal(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmb->sumDecimal();
        $this->assertEquals(
            55.0,
            $t1->toFloat()
        );

        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $enmb2->sumDecimal(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            15.0,
            $t2->toFloat()
        );
    }

    public function testAny(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        $this->assertTrue(
            $enmb->any()
        );
        $this->assertTrue(
            $enmb->any(function ($item) {
                return $item > 2;
            })
        );
        $this->assertTrue(
            $enmb->any(function ($item) {
                return $item === 0;
            })
        );
        $this->assertFalse(
            $enmb->any(function ($item) {
                return $item > 15;
            })
        );
        $enmb = Enumerable::fromArray([]);
        $this->assertFalse(
            $enmb->any()
        );

    }

    public function testAll(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);


        $this->assertTrue(
            $enmb->all(function ($item) {
                return $item < 20;
            })
        );
        $this->assertTrue(
            $enmb->all(function ($item) {
                return is_int($item);
            })
        );
        $this->assertFalse(
            $enmb->all(function ($item) {
                return $item > 5;
            })
        );

        $emptyEnmb = Enumerable::fromArray([]);

        $this->assertTrue(
            $emptyEnmb->all(function ($item) {
                return $item > 5;
            })
        );

    }

    public function testAverage(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmb->average();
        $this->assertEquals(
            5.0,
            $t1
        );

        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $enmb2->average(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            2.5,
            $t2
        );
    }

    public function testAverageDecimal(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmb->averageDecimal();
        $this->assertEquals(
            5.0,
            $t1->toFloat()
        );

        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $enmb2->averageDecimal(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            2.5,
            $t2->toFloat()
        );

        $enmb = Enumerable::fromArray(
            [
                Decimal::wrap(0),
                Decimal::wrap(1),
                Decimal::wrap(2),
                Decimal::wrap(3),
                Decimal::wrap(4),
                Decimal::wrap(5),
                Decimal::wrap(6),
                Decimal::wrap(7),
                Decimal::wrap(8),
                Decimal::wrap(9),
                Decimal::wrap(10),
            ]
        );
        $t1 = $enmb->averageDecimal(null,4,".");
        $this->assertEquals(
            "5.0000",
            $t1->getValue()
        );

        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $enmb2->averageDecimal(function (MockRefObj $item) {
            return $item->propA;
        },6,".");
        $this->assertEquals(
            "2.500000",
            $t2->getValue()
        );

        $enmb = Enumerable::fromArray(
            [
               "0",
                "1.000,10",
                "2.000,10",
                "3.000,10",
                "4.000,10",
                "5.000,10",
                "6.000,10",
                "7.000,10",
                "8.000,10",
                "9.000,10",
                "10.000,10",
            ]
        );
        $t1 = $enmb->averageDecimal(null,4,",");
        $this->assertEquals(
            "5000.0909",
            $t1->getValue()
        );
        $this->assertEquals(
            5000.0909,
            $t1->toFloat()
        );
    }

    public function testMin(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmb->min();
        $this->assertEquals(
            0,
            $t1
        );

        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $enmb2->min(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            0,
            $t2
        );
    }

    public function testMinDecimal(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmb->minDecimal();
        $this->assertEquals(
            0,
            $t1->toFloat()
        );

        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $enmb2->minDecimal(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            0,
            $t2->toFloat()
        );

        $enmb = Enumerable::fromArray(
            [
                Decimal::wrap(0),
                Decimal::wrap(1),
                Decimal::wrap(2),
                Decimal::wrap(3),
                Decimal::wrap(4),
                Decimal::wrap(5),
                Decimal::wrap(6),
                Decimal::wrap(7),
                Decimal::wrap(8),
                Decimal::wrap(9),
                Decimal::wrap(10),
            ]
        );
        $t1 = $enmb->minDecimal(null,4,".");
        $this->assertEquals(
            "0.0000",
            $t1->getValue()
        );

        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $enmb2->minDecimal(function (MockRefObj $item) {
            return ($item->propA + 2 ) * $item->propB;
        },6,".");
        $this->assertEquals(
            "10.000000",
            $t2->getValue()
        );

        $enmb = Enumerable::fromArray(
            [
                "1.000,10",
                "2.000,10",
                "3.000,10",
                "4.000,10",
                "5.000,10",
                "6.000,10",
                "7.000,10",
                "8.000,10",
                "9.000,10",
                "10.000,10",
            ]
        );
        $t1 = $enmb->minDecimal(null,4,",");
        $this->assertEquals(
            "1000.1000",
            $t1->getValue()
        );
        $this->assertEquals(
            1000.1,
            $t1->toFloat()
        );

    }

    public function testMax(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 25, 6, 7, 8, 9, 10]);
        $t1 = $enmb->max();
        $this->assertEquals(
            25,
            $t1
        );

        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(25, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $enmb2->max(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            25,
            $t2
        );
    }


    public function testMaxDecimal(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 25, 6, 7, 8, 9, 10]);
        $t1 = $enmb->maxDecimal();
        $this->assertEquals(
            "25.0000000000",
            $t1->getValue()
        );

        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(25, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $enmb2->maxDecimal(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            25.0,
            $t2->toFloat()
        );
    }

    public function testReverse(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5]);
        $t1 = $enmb->reverse();
        $this->assertEquals(
            [5, 4, 3, 2, 1, 0],
            $t1->toArray()
        );
    }

    public function testAsEnumarable(): void
    {
        $lst = new FixedArrayList([0, 2, 3, 4, 5]);
        $t1 = $lst->where(function (int $item) {
            return $item > 3;
        });

        $t2 = $lst->where(function (int $item) {
            return $item > 3;
        })->asEnumerable();

        $lst->add(8);

        $this->assertEquals(
            [4, 5, 8],
            $t1->toArray()
        );
        $this->assertEquals(
            [4, 5],
            $t2->toArray()
        );


    }

    public function testSort(): void
    {
        $enmr = Enumerable::fromArray([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $enmr->sort();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t1->toArray()
        );
        $t2 = $enmr->sort(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t2->toArray()
        );
    }

    public function testSortDesc(): void
    {
        $enmr = Enumerable::fromArray([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $enmr->sortDesc();
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t1->toArray()
        );
        $t2 = $enmr->sortDesc(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t2->toArray()
        );
    }

    public function testSortKey(): void
    {
        $enmr = Enumerable::fromArray(
            [
                'a' => 3,
                'c' => 8,
                'f' => 4,
                'd' => 1,
                'g' => 7,
                'e' => 0,
                'b' => 9,
                'h' => 5,
                'z' => 2,
                'k' => 6
            ]);
        $t1 = $enmr->sortKey();
        $this->assertEquals(
            [
                'a' => 3,
                'b' => 9,
                'c' => 8,
                'd' => 1,
                'e' => 0,
                'f' => 4,
                'g' => 7,
                'h' => 5,
                'k' => 6,
                'z' => 2,
            ],
            $t1->toArray(true)
        );
        $t2 = $enmr->sortKey(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [
                'z' => 2,
                'k' => 6,
                'h' => 5,
                'g' => 7,
                'f' => 4,
                'e' => 0,
                'd' => 1,
                'c' => 8,
                'b' => 9,
                'a' => 3,
            ],
            $t2->toArray(true)
        );
    }

    public function testSortKeyDesc(): void
    {
        $enmr = Enumerable::fromArray(
            [
                'a' => 3,
                'c' => 8,
                'f' => 4,
                'd' => 1,
                'g' => 7,
                'e' => 0,
                'b' => 9,
                'h' => 5,
                'z' => 2,
                'k' => 6
            ]);
        $t1 = $enmr->sortKeyDesc();
        $this->assertEquals(
            [
                'z' => 2,
                'k' => 6,
                'h' => 5,
                'g' => 7,
                'f' => 4,
                'e' => 0,
                'd' => 1,
                'c' => 8,
                'b' => 9,
                'a' => 3,
            ]
            ,
            $t1->toArray(true)
        );
        $t2 = $enmr->sortKeyDesc(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [
                'a' => 3,
                'b' => 9,
                'c' => 8,
                'd' => 1,
                'e' => 0,
                'f' => 4,
                'g' => 7,
                'h' => 5,
                'k' => 6,
                'z' => 2,
            ]
            ,
            $t2->toArray(true)
        );
    }

    public function testOrderBy(): void
    {
        $enmr = Enumerable::fromArray([3, 8, 4, 1, 7, 0, 9, 5, 2, 6])
        ->select(fn($i) => new Decimal($i,2));
        $t0 = $enmr->orderBy();
        $this->assertEquals(
            ["0.00", "1.00", "2.00", "3.00", "4.00", "5.00",
                "6.00", "7.00", "8.00", "9.00"],
            $t0->select(fn(Decimal $i) => $i->getValue())
                ->toArray()
        );

        $enmr = Enumerable::fromArray([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $enmr->orderBy();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t1->toArray()
        );
        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 3),
            new MockRefObj(1, 8),
            new MockRefObj(2, 1),
            new MockRefObj(3, 5),
            new MockRefObj(4, 6),
            new MockRefObj(5, 2),
        ]);
        $t2 = $enmb2->orderBy(function (MockRefObj $item) {
            return $item->propB;
        });
        $this->assertEquals(
            [
                new MockRefObj(2, 1),
                new MockRefObj(5, 2),
                new MockRefObj(0, 3),
                new MockRefObj(3, 5),
                new MockRefObj(4, 6),
                new MockRefObj(1, 8),
            ],
            $t2->toArray()
        );

        $t3 = $enmb2->orderBy(function (MockRefObj $item) {
            return $item->propB + $item->propA;
        });
        $this->assertEquals(
            [
                new MockRefObj(0, 3), // 3
                new MockRefObj(2, 1), // 3
                new MockRefObj(5, 2), // 7
                new MockRefObj(3, 5), // 8
                new MockRefObj(1, 8), // 9
                new MockRefObj(4, 6), // 10

            ],
            $t3->toArray()
        );

    }

    public function testOrderByDesc(): void
    {
        $enmr = Enumerable::fromArray([3, 8, 4, 1, 7, 0, 9, 5, 2, 6])
            ->select(fn($i) => new Decimal($i,4));
        $t0 = $enmr->orderByDesc();
        $this->assertEquals(
            ["9.0000", "8.0000", "7.0000", "6.0000", "5.0000", "4.0000", "3.0000", "2.0000", "1.0000", "0.0000"],
            $t0->select(fn(Decimal $i) => $i->getValue())
                ->toArray()
        );
        $enmr = Enumerable::fromArray([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $enmr->orderByDesc();
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t1->toArray()
        );
        $enmb2 = Enumerable::fromArray([
            new MockRefObj(0, 3),
            new MockRefObj(1, 8),
            new MockRefObj(2, 1),
            new MockRefObj(3, 5),
            new MockRefObj(4, 6),
            new MockRefObj(5, 2),
        ]);
        $t2 = $enmb2->orderByDesc(function (MockRefObj $item) {
            return $item->propB;
        });
        $this->assertEquals(
            [
                new MockRefObj(1, 8),
                new MockRefObj(4, 6),
                new MockRefObj(3, 5),
                new MockRefObj(0, 3),
                new MockRefObj(5, 2),
                new MockRefObj(2, 1),
            ],
            $t2->toArray()
        );

        $t3 = $enmb2->orderByDesc(function (MockRefObj $item) {
            return $item->propB + $item->propA;
        });
        $this->assertEquals(
            [
                new MockRefObj(4, 6), // 10
                new MockRefObj(1, 8), // 9
                new MockRefObj(3, 5), // 8
                new MockRefObj(5, 2), // 7
                new MockRefObj(2, 1), // 3
                new MockRefObj(0, 3), // 3
            ],
            $t3->toArray()
        );

    }

    public function testConcat(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5]);
        $t1 = $enmb->concat([6, 7, 8, 9, 10]);
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
            $t1->toArray()
        );
    }

    public function testDistinct(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 0, 1, 1, 0]);
        $t1 = $enmb->distinct();
        $this->assertEquals(
            [0, 1],
            $t1->toArray()
        );

        $enmb = Enumerable::fromArray([
            new MockRefObj(1, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 4),
            new MockRefObj(3, 6),
            new MockRefObj(2, 4),

        ]);
        $t2 = $enmb->distinct();
        $this->assertEquals(
            [
                new MockRefObj(1, 5),
                new MockRefObj(2, 4),
                new MockRefObj(3, 6),
            ],
            $t2->toArray()
        );

        $a = new MockRefObj(1, 5);
        $b = new MockRefObj(1, 5);
        $c = new MockRefObj(1, 5);

        $enmb = Enumerable::fromArray([
            new MockRefObj(1, 5),
            $c,
            new MockRefObj(1, 5),
            $a,
            $b,
        ]);
        $t3 = $enmb->distinct();
        $this->assertEquals(
            [
                new MockRefObj(1, 5),
            ],
            $t3->toArray()
        );

        $enmb = Enumerable::fromArray([
            $a,
            $c,
            $b,
            $a,
            $b,
            $c,
        ]);
        $t4 = $enmb->distinct(function ($a, $b) {
            return $a === $b;
        });
        $this->assertEquals(
            [
                $a,
                $c,
                $b,
            ],
            $t4->toArray()
        );


    }

    public function testExcept(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmb->except([10, 9, 5, 6, 7, 8]);
        $this->assertEquals(
            [0, 1, 2, 3, 4],
            $t1->toArray()
        );
    }

    public function testFirstOrNUll(): void
    {
        $enmb = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $this->assertEquals(
            0,
            $enmb->firstOrNull()
        );
        $this->assertEquals(
            6,
            $enmb->firstOrNull(function ($item) {
                return $item > 5;
            })
        );
        $this->assertEquals(
            null,
            $enmb->firstOrNull(function ($item) {
                return $item > 15;
            })
        );
    }

    public function testAggregate(): void
    {
        $enmb = Enumerable::fromArray([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $this->assertEquals(
            3628800,
            $enmb->aggregate(function (&$total, $item) {
                $total *= $item;
            })
        );
        $this->assertEquals(
            181440,
            $enmb->aggregate(
                function (&$total, $item) {
                    $total *= $item;
                },
                null,
                function ($total, Enumerable $th) {
                    return ($total / $th->count()) / 2;
                })
        );
        $this->assertEquals(
            3.25,
            $enmb->aggregate(
                function (&$total, $item) {
                    $total += $item;
                },
                10,
                function ($total, Enumerable $th) {
                    return ($total / $th->count()) / 2;
                })
        );
    }

    public function testUnion(): void
    {
        $enmr = Enumerable::fromArray([-3, '1', 1, 2, 3, 3, 4, 4, 5]);
        $t1 = $enmr->union([3, 8, 4, 8]);

        $this->assertEquals(
            [-3, 1, 2, 3, 4, 5, 8],
            $t1->toArray()
        );
        $t2 = $enmr->union([3, 8, 4, 8], function ($a, $b) {
            return $a === $b;
        });
        $this->assertEquals(
            [-3, '1', 1, 2, 3, 4, 5, 8],
            $t2->toArray()
        );
    }

    public function testIntersect(): void
    {
        $enmr = Enumerable::fromArray([1, 3, 4, 5]);
        $t1 = $enmr->intersect([3, 8, 4]);

        $this->assertEquals(
            [3, 4],
            $t1->toArray()
        );
        $t2 = $enmr->intersect([]);

        $this->assertEquals(
            [],
            $t2->toArray()
        );

        $a = new MockRefObj(1, 5);
        $b = new MockRefObj(1, 5);
        $c = new MockRefObj(1, 5);
        $d = new MockRefObj(1, 5);
        $e = new MockRefObj(1, 5);
        $f = new MockRefObj(1, 5);

        $enmr2 = Enumerable::fromArray([$a, $b, $c, $d]);
        $t3 = $enmr2->intersect([$f, $d, $e, $b], function ($first, $second) {
            return $first === $second;
        });

        $this->assertEquals(
            [$b, $d],
            $t3->toArray()
        );

    }

    public function testZip(): void
    {
        $enmr = Enumerable::fromArray([1, 2, 3, 4, 5]);
        $t1 = $enmr->zip(['a', 'b', 'c', 'd', 'e'], function ($f, $s) {
            return $f . '-' . $s;
        });

        $this->assertEquals(
            ['1-a', '2-b', '3-c', '4-d', '5-e'],
            $t1->toArray()
        );

        $t2 = $enmr->zip(['a', 'b', 'c'], function ($f, $s) {
            return $f . '-' . $s;
        });

        $this->assertEquals(
            ['1-a', '2-b', '3-c'],
            $t2->toArray()
        );

        $enmr2 = Enumerable::fromArray([1, 2]);
        $t3 = $enmr2->zip(['a', 'b', 'c', 'd', 'e'], function ($f, $s) {
            return $f . '-' . $s;
        });

        $this->assertEquals(
            ['1-a', '2-b'],
            $t3->toArray()
        );
    }

    public function testLastOrNull(): void
    {
        $enmr = Enumerable::fromArray([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmr->lastOrNull();
        $this->assertEquals(
            10,
            $t1
        );
        $t2 = $enmr->lastOrNull(function ($item) {
            return $item < 5;
        });
        $this->assertEquals(
            4,
            $t2
        );
    }

    public function testFromIterable(): void
    {
        // array test
        $enmr = Enumerable::fromIterable([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmr->toArray();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
            $t1
        );

        // generator test
        $enmr = Enumerable::fromIterable((function ($start, $step, $count) {
            $cur = $start;
            for ($i = 0; $i < $count; $i++) {
                yield $cur;
                $cur += $step;
            }
        })(0, 2, 5));
        $t1 = $enmr->toArray();
        $this->assertEquals(
            [0, 2, 4, 6, 8],
            $t1
        );

        // IteratorAggregate test
        $enmr = Enumerable::fromIterable(new ArrayList([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]));
        $t1 = $enmr->toArray();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
            $t1
        );

        // Iteratortest
        $splArr = new SplFixedArray(11);
        for ($i = 0; $i < 11; $i++) {
            $splArr[$i] = $i;
        }
        $enmr = Enumerable::fromIterable($splArr);
        $t1 = $enmr->toArray();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
            $t1
        );

    }

    public function testToGroupedArray(): void
    {
        $testArr = [
            (object)['id' => 12, 'pId' => 1, "name" => "a"],
            (object)['id' => 8, 'pId' => 3, "name" => "d"],
            (object)['id' => 10, 'pId' => 1, "name" => "b"],
            (object)['id' => 2, 'pId' => 3, "name" => "e"],
            (object)['id' => 4, 'pId' => 2, "name" => "f"],
            (object)['id' => 35, 'pId' => 4, "name" => "t"],
            (object)['id' => 6, 'pId' => 2, "name" => "g"],
            (object)['id' => 52, 'pId' => 2, "name" => "h"],
            (object)['id' => 50, 'pId' => 2, "name" => "h"],

        ];
        $res = Enumerable::fromArray($testArr)
            ->toGroupedArray(function ($i) {
                return $i->pId;
            });
        $this->assertEquals(
            [
                1 => [
                    (object)['id' => 12, 'pId' => 1, "name" => "a"],
                    (object)['id' => 10, 'pId' => 1, "name" => "b"],
                ],
                3 => [
                    (object)['id' => 8, 'pId' => 3, "name" => "d"],
                    (object)['id' => 2, 'pId' => 3, "name" => "e"],
                ],
                2 => [
                    (object)['id' => 4, 'pId' => 2, "name" => "f"],
                    (object)['id' => 6, 'pId' => 2, "name" => "g"],
                    (object)['id' => 52, 'pId' => 2, "name" => "h"],
                    (object)['id' => 50, 'pId' => 2, "name" => "h"],
                ],
                4 => [
                    (object)['id' => 35, 'pId' => 4, "name" => "t"],
                ]
            ],
            $res
        );

        $res = Enumerable::fromArray($testArr)
            ->toGroupedArray(function ($i) {
                return $i->pId;
            }, function ($i) {
                return $i->name;
            });
        $this->assertEquals(
            [
                1 => [
                    "a",
                    "b",
                ],
                3 => [
                    "d",
                    "e",
                ],
                2 => [
                    "f",
                    "g",
                    "h",
                    "h",
                ],
                4 => [
                    "t",
                ]
            ],
            $res
        );
    }

    public function testToGroupedDictionary(): void
    {
        $testArr = [
            (object)['id' => 12, 'pId' => 1, "name" => "a"],
            (object)['id' => 8, 'pId' => 3, "name" => "d"],
            (object)['id' => 10, 'pId' => 1, "name" => "b"],
            (object)['id' => 2, 'pId' => 3, "name" => "e"],
            (object)['id' => 4, 'pId' => 2, "name" => "f"],
            (object)['id' => 35, 'pId' => 4, "name" => "t"],
            (object)['id' => 6, 'pId' => 2, "name" => "g"],
            (object)['id' => 52, 'pId' => 2, "name" => "h"],
            (object)['id' => 50, 'pId' => 2, "name" => "h"],

        ];
        $res = Enumerable::fromArray($testArr)
            ->toGroupedDictionary(function ($i) {
                return $i->pId;
            });
        $this->assertTrue($res instanceof Dictionary);
        $this->assertTrue($res[1] instanceof ArrayList);
        $this->assertEquals(
            [
                1 => [
                    (object)['id' => 12, 'pId' => 1, "name" => "a"],
                    (object)['id' => 10, 'pId' => 1, "name" => "b"],
                ],
                3 => [
                    (object)['id' => 8, 'pId' => 3, "name" => "d"],
                    (object)['id' => 2, 'pId' => 3, "name" => "e"],
                ],
                2 => [
                    (object)['id' => 4, 'pId' => 2, "name" => "f"],
                    (object)['id' => 6, 'pId' => 2, "name" => "g"],
                    (object)['id' => 52, 'pId' => 2, "name" => "h"],
                    (object)['id' => 50, 'pId' => 2, "name" => "h"],
                ],
                4 => [
                    (object)['id' => 35, 'pId' => 4, "name" => "t"],
                ]
            ],
            $res->select(function (ArrayList $i) {
                return $i->toArray();
            })->toArray(true)
        );

        $res = Enumerable::fromArray($testArr)
            ->toGroupedDictionary(function ($i) {
                return $i->pId;
            }, function ($i) {
                return $i->name;
            });
        $this->assertTrue($res instanceof Dictionary);
        $this->assertTrue($res[1] instanceof ArrayList);
        $this->assertEquals(
            [
                1 => [
                    "a",
                    "b",
                ],
                3 => [
                    "d",
                    "e",
                ],
                2 => [
                    "f",
                    "g",
                    "h",
                    "h",
                ],
                4 => [
                    "t",
                ]
            ],
            $res->select(function (ArrayList $i) {
                return $i->toArray();
            })->toArray(true)
        );
    }

}
