<?php


namespace Gek\Collections\Tests;


use ArrayIterator;
use ArrayObject;
use Gek\Collections\Dictionary;
use Gek\Collections\Enumerable;
use Gek\Collections\IEnumerable;
use Gek\Collections\Iterators\DictionaryIterator;
use Gek\Collections\Tests\Fixtures\MockRefObj;
use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Iterator;
use PHPUnit\Framework\TestCase;

/**
 * Class DictionaryTest
 * @package Gek\Collections\Tests
 */
class DictionaryTest extends TestCase
{
    public function testConstruct(): void
    {

        // array
        $arr = array('a' => 10, 'b' => 15, 'c' => 16, 'd' => 17);
        $lst = new Dictionary($arr);
        $this->assertEquals(
            count($arr),
            $lst->count()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray(true)
        );

        //Enumerable
        $enmr = Enumerable::fromArray($arr);
        $lst = new Dictionary($enmr);
        $this->assertEquals(
            $enmr->count(),
            $lst->count()
        );
        $this->assertEquals(
            $enmr->toArray(true),
            $lst->toArray(true)
        );

        // IteratorAggregate
        $arrObj = new ArrayObject($arr);
        $lst = new Dictionary($arrObj);
        $this->assertEquals(
            $arrObj->count(),
            $lst->count()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray(true)
        );

        // Iterator
        $itr = new ArrayIterator($arr);
        $lst = new Dictionary($itr);
        $this->assertEquals(
            $itr->count(),
            $lst->count()
        );
        $this->assertEquals(
            iterator_to_array($itr, true),
            $lst->toArray(true)
        );

        // generator
        $generator = function () {
            for ($i = 0; $i < 5; $i++) {
                yield 'a' . $i => $i;
            }
        };
        $lst = new Dictionary($generator());
        $this->assertEquals(
            5,
            $lst->count()
        );
        $this->assertEquals(
            iterator_to_array($generator(), true),
            $lst->toArray(true)
        );

        // object
        $obj = (object)[
            'propA' => 'a',
            'propB' => 'b',
        ];
        $lst = new Dictionary($obj);
        $this->assertEquals(
            2,
            $lst->count()
        );
        $this->assertEquals(
            [
                'propA' => 'a',
                'propB' => 'b'
            ],
            $lst->toArray(true)
        );

        // class
        $cls = new MockRefObj(3, 5);
        $lst = new Dictionary($cls);
        $this->assertEquals(
            2,
            $lst->count()
        );
        $this->assertEquals(
            ['propA' => 3, 'propB' => 5],
            $lst->toArray(true)
        );
    }

    #region ArrayAccess

    public function testOffsetExists(): void
    {
        $arr = [
            'a' => 0,
            'b' => 1,
            'c' => 2,
            'd' => 3,
            'e' => 4,
            'f' => 5,
            'g' => 6,
            'h' => 7,
        ];
        $lst = new Dictionary($arr);
        $this->assertTrue(
            $lst->offsetExists('f')
        );
        // php isset
        $this->assertTrue(
            isset($lst['f'])
        );
        $this->assertFalse(
            $lst->offsetExists('z')
        );
        // php isset
        $this->assertFalse(
            isset($lst['z'])
        );
    }

    public function testOffsetGet(): void
    {
        $arr = [
            'a' => 0,
            'b' => 1,
            'c' => 2,
            'd' => 3,
            'e' => 4,
            'f' => 5,
            'g' => 6,
            'h' => 7,
        ];
        $lst = new Dictionary($arr);
        $this->assertEquals(
            5,
            $lst->offsetGet('f')
        );
        // php []
        $this->assertEquals(
            5,
            $lst['f']
        );
        $this->expectException(ArgumentOutOfRangeException::class);
        $lst['z'];
    }

    public function testOffsetSet(): void
    {
        $arr = [
            'a' => 0,
            'b' => 1,
            'c' => 2,
            'd' => 3,
            'e' => 4,
            'f' => 5,
            'g' => 6,
            'h' => 7,
        ];
        $lst = new Dictionary($arr);
        $lst->offsetSet('f', 'beş');
        $this->assertEquals(
            'beş',
            $lst['f']
        );
        // php [] set
        $lst['g'] = 'altı';
        $this->assertEquals(
            'altı',
            $lst['g']
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst['zz'] = 'invalid index ("zz") set';

    }

    public function testOffsetUnset(): void
    {
        $arr = [
            'a' => 0,
            'b' => 1,
            'c' => 2,
            'd' => 3,
            'e' => 4,
            'f' => 5,
            'g' => 6,
            'h' => 7,
        ];
        $lst = new Dictionary($arr);

        $lst->offsetUnset('f');
        $this->assertEquals(
            7,
            $lst->count()
        );

        //php unset
        unset($lst['g']);
        $this->assertEquals(
            6,
            $lst->count()
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        unset($lst['zz']);

    }

    #endregion ArrayAccess

    #region IEnumerable

    public function testGetIterator(): void
    {
        $lst = new Dictionary();
        $this->assertInstanceOf(
            Iterator::class,
            $lst->getIterator()
        );
        $this->assertInstanceOf(
            DictionaryIterator::class,
            $lst->getIterator()
        );
    }

    public function testCount(): void
    {
        $lst = new Dictionary(range(0, 10));
        $this->assertEquals(
            11,
            $lst->count()
        );
        $this->assertEquals(
            11,
            count($lst)
        );
    }

    #endregion IEnumerable

    #region ICollection

    public function testAdd(): void
    {
        $lst = new Dictionary();
        $this->assertEquals(
            0,
            $lst->count()
        );
        $lst->add('test');
        $this->assertEquals(
            1,
            $lst->count()
        );

        $this->assertEquals(
            'test',
            $lst[0]
        );

        $lst->add('test2');
        $this->assertEquals(
            2,
            $lst->count()
        );

        $this->assertEquals(
            'test2',
            $lst[1]
        );
        $this->assertEquals(
            [
                0 => 'test',
                1 => 'test2'
            ],
            $lst->toArray(true)
        );


    }


    public function testAddRange(): void
    {
        $arr = [
            'a' => 0,
            'b' => 1,
            'c' => 2,
            'd' => 3,
        ];
        $lst = new Dictionary($arr);

        $this->assertEquals(
            4,
            $lst->count()
        );

        $lst->addRange([
            'e' => 4,
            'f' => 5,
            'g' => 6,
            'h' => 7,
        ]);
        $this->assertEquals(
            8,
            $lst->count()
        );

        $this->assertEquals(
            [
                'a' => 0,
                'b' => 1,
                'c' => 2,
                'd' => 3,
                'e' => 4,
                'f' => 5,
                'g' => 6,
                'h' => 7,
            ],
            $lst->toArray(true)
        );
        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->addRange(
            [
                'k' => 4564,
                'e' => 446,
                'z' => 45688
            ]
        );

    }

    public function testClear(): void
    {
        $lst = new Dictionary(range(0, 10));

        $this->assertEquals(
            11,
            $lst->count()
        );
        $this->assertEquals(
            true,
            $lst->any()
        );

        $lst->clear();

        $this->assertEquals(
            0,
            $lst->count()
        );
        $this->assertEquals(
            false,
            $lst->any()
        );

    }

    public function testContains(): void
    {
        $lst = new Dictionary();

        $stdObj = (object)[
            'prop' => 'a'
        ];
        $lst->add($stdObj);

        $arr = array(
            'a', 'b', 'c', 'd'
        );
        $lst->add($arr);

        $vlInt = 5;
        $lst->add($vlInt);

        $vlFloat = 15.4564;
        $lst->add($vlFloat);

        $refObj = new MockRefObj(3, 5);
        $lst->add($refObj);

        $vlStr = 'str-value';
        $lst->add($vlStr);

        $this->assertEquals(
            true,
            $lst->contains($stdObj)
        );
        $stdObj2 = (object)[
            'prop' => 'a',
        ];
        $this->assertEquals(
            true,
            $lst->contains($stdObj2)
        );
        $stdObj->prop = 'b';
        $this->assertEquals(
            false,
            $lst->contains($stdObj2)
        );
        $arr2 = array(
            'a', 'b', 'c', 'd'
        );
        // var_dump($lst->toArray());
        $this->assertEquals(
            true,
            $lst->contains($arr)
        );
        $this->assertEquals(
            true,
            $lst->contains($arr2)
        );
        $arr[] = 'e';
        $this->assertEquals(
            false,
            $lst->contains($arr)
        );

        $this->assertEquals(
            true,
            $lst->contains($refObj)
        );
        $refObj->propA = 10;
        $this->assertEquals(
            true,
            $lst->contains($refObj)
        );
        $refObj2 = $refObj;
        $this->assertEquals(
            true,
            $lst->contains($refObj2)
        );

        $refObj2->propB = 25;
        $this->assertEquals(
            true,
            $lst->contains($refObj2)
        );

        $refObj2 = clone $refObj;
        $refObj2->propB = 40;
        $this->assertEquals(
            false,
            $lst->contains($refObj2)
        );
        $this->assertEquals(
            true,
            $lst->contains(15.4564)
        );
        $this->assertEquals(
            true,
            $lst->contains($vlInt)
        );

        $this->assertEquals(
            true,
            $lst->contains('str-value')
        );


    }


    public function testRemove(): void
    {
        $arr = [
            'a' => 1,
            'b' => 2,
            'c' => 3,
            'd' => 4,
            'e' => 5,
            'f' => 6,
        ];
        $lst = new Dictionary($arr);
        $this->assertEquals(
            6,
            $lst->count()
        );
        $lst->remove(2);
        $this->assertEquals(
            5,
            $lst->count()
        );


        $lst->remove(4);
        $this->assertEquals(
            4,
            $lst->count()
        );
        $this->assertEquals(
            [
                'a' => 1,
                'c' => 3,
                'e' => 5,
                'f' => 6,
            ],
            $lst->toArray(true)
        );
    }

    #endregion ICollection

    #region IDictionary

    public function testAddKeyValue(): void
    {

        $lst = new Dictionary();
        $this->assertEquals(
            0,
            $lst->count()
        );
        $lst->addKeyValue('key1', 'test');
        $this->assertEquals(
            1,
            $lst->count()
        );

        $this->assertEquals(
            'test',
            $lst['key1']
        );

        $lst->addKeyValue('key2', 'test2');
        $this->assertEquals(
            2,
            $lst->count()
        );

        $this->assertEquals(
            'test2',
            $lst['key2']
        );
        $this->assertEquals(
            [
                'key1' => 'test',
                'key2' => 'test2'
            ],
            $lst->toArray(true)
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->addKeyValue('key1', 'val');

    }

    public function testTryAddKeyValue(): void
    {

        $lst = new Dictionary();
        $this->assertEquals(
            0,
            $lst->count()
        );
        $this->assertTrue(
            $lst->tryAddKeyValue('key1', 'test')
        );

        $this->assertEquals(
            1,
            $lst->count()
        );

        $this->assertEquals(
            'test',
            $lst['key1']
        );

        $this->assertTrue(
            $lst->tryAddKeyValue('key2', 'test2')
        );
        $this->assertEquals(
            2,
            $lst->count()
        );

        $this->assertEquals(
            'test2',
            $lst['key2']
        );
        $this->assertEquals(
            [
                'key1' => 'test',
                'key2' => 'test2'
            ],
            $lst->toArray(true)
        );

        $this->assertFalse($lst->tryAddKeyValue('key1', 'val'));
    }

    public function testContainsKey(): void
    {
        $arr = [
            'a' => 1,
            'b' => 2,
            'c' => 3,
            'd' => 4,
            'e' => 5,
            'f' => 6
        ];

        $lst = new Dictionary($arr);

        $this->assertTrue(
            $lst->containsKey('d')
        );

        $this->assertTrue(
            $lst->containsKey('a')
        );

        $this->assertTrue(
            $lst->containsKey('f')
        );

        $this->assertFalse(
            $lst->containsKey('zzz')
        );
    }

    public function testRemoveByKey(): void
    {
        $arr = [
            'a' => 1,
            'b' => 2,
            'c' => 3,
            'd' => 4,
            'e' => 5,
            'f' => 6
        ];
        $lst = new Dictionary($arr);

        $this->assertTrue(
            $lst->removeByKey('d')
        );

        $this->assertEquals(
            5,
            $lst->count()
        );

        $this->assertEquals(
            [
                'a' => 1,
                'b' => 2,
                'c' => 3,
                'e' => 5,
                'f' => 6,
            ],
            $lst->toArray(true)
        );

        $this->assertFalse(
            $lst->removeByKey('zz')
        );
    }

    #endregion IDictionary

    #region Enumerable

    public function testWhere(): void
    {
        $lst = new Dictionary([
            'a' => 10, 'b' => 25, 'c' => 18, 'd' => 26, 'e' => 21, 'f' => 30, 'g' => 20, 'h' => 48, 'k' => 5
        ]);
        $t1 = $lst->where(function ($item, $key) {
            return $item > 20 && $key != 'e';
        })->toArray(true);
        $this->assertEquals(
            ['b' => 25, 'd' => 26, 'f' => 30, 'h' => 48],
            $t1
        );
        $t2 = $lst->where(function ($item, $key) {
            return $item < 20 && $key != 'a';
        });
        $this->assertInstanceOf(
            IEnumerable::class,
            $t2
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t2
        );
        $lst->addKeyValue('m', 8);
        $this->assertEquals(
            [
                'c' => 18, 'k' => 5, 'm' => 8
            ],
            $t2->toArray(true)
        );

    }

    public function testLimit(): void
    {
        $lst = new Dictionary([
            'a' => 10, 'b' => 25, 'c' => 18, 'd' => 26, 'e' => 21, 'f' => 30, 'g' => 20, 'h' => 48, 'k' => 5
        ]);
        $t1 = $lst->limit(2, 2)->toArray(true);
        $this->assertEquals(
            [
                'c' => 18, 'd' => 26
            ],
            $t1
        );
        $t2 = $lst->limit(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t2
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t2
        );
        $lst->addKeyValue('m', 8);
        $lst->addKeyValue('z', 7);
        $this->assertEquals(
            [
                'f' => 30, 'g' => 20, 'h' => 48, 'k' => 5,
                'm' => 8, 'z' => 7
            ],
            $t2->toArray(true)
        );

        $t3 = $lst->limit(50, 10)->toArray(true);
        $this->assertTrue(empty($t3));
    }

    public function testSkip(): void
    {
        $lst = new Dictionary([
            'a' => 10, 'b' => 25, 'c' => 18, 'd' => 26, 'e' => 21, 'f' => 30, 'g' => 20, 'h' => 48, 'k' => 5
        ]);
        $t1 = $lst->skip(0)->toArray(true);
        $this->assertEquals(
            [
                'a' => 10, 'b' => 25, 'c' => 18, 'd' => 26, 'e' => 21, 'f' => 30, 'g' => 20, 'h' => 48, 'k' => 5
            ],
            $t1
        );
        $t2 = $lst->skip(5)->toArray(true);
        $this->assertEquals(
            [
                'f' => 30, 'g' => 20, 'h' => 48, 'k' => 5
            ],
            $t2
        );

        $t3 = $lst->skip(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t3
        );
        $lst->addKeyValue('m', 11);
        $this->assertInstanceOf(
            Enumerable::class,
            $t3
        );
        $this->assertEquals(
            [
                'f' => 30, 'g' => 20, 'h' => 48, 'k' => 5, 'm' => 11
            ],
            $t3->toArray(true)
        );

    }

    public function testTake(): void
    {
        $lst = new Dictionary([
            'a' => 10, 'b' => 25, 'c' => 18, 'd' => 26, 'e' => 21, 'f' => 30, 'g' => 20, 'h' => 48, 'k' => 5
        ]);
        $t1 = $lst->take(5)->toArray(true);
        $this->assertEquals(
            [
                'a' => 10, 'b' => 25, 'c' => 18, 'd' => 26, 'e' => 21,
            ],
            $t1
        );
        $t2 = $lst->take(20)->toArray(true);
        $this->assertEquals(
            [
                'a' => 10, 'b' => 25, 'c' => 18, 'd' => 26, 'e' => 21, 'f' => 30, 'g' => 20, 'h' => 48, 'k' => 5
            ],
            $t2
        );

        $t3 = $lst->take(20);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t3
        );
        $lst->addKeyValue('m', 11);
        $this->assertInstanceOf(
            Enumerable::class,
            $t3
        );
        $this->assertEquals(
            [
                'a' => 10, 'b' => 25, 'c' => 18, 'd' => 26, 'e' => 21, 'f' => 30, 'g' => 20, 'h' => 48, 'k' => 5,
                'm' => 11
            ],
            $t3->toArray(true)
        );

    }

    public function testSelect(): void
    {
        $lst = new Dictionary([
            'a' => 10, 'b' => 25, 'c' => 18, 'd' => 26, 'e' => 21, 'f' => 30, 'g' => 20, 'h' => 48, 'k' => 5
        ]);
        $t1 = $lst->select(function ($item, $key) {
            return $item . $key;
        })->toArray(true);
        $this->assertEquals(
            [
                'a' => '10a',
                'b' => '25b',
                'c' => '18c',
                'd' => '26d',
                'e' => '21e',
                'f' => '30f',
                'g' => '20g',
                'h' => '48h',
                'k' => '5k'
            ],
            $t1
        );

        $lst = new Dictionary([
            'a' => new MockRefObj(0, 5),
            'b' => new MockRefObj(1, 5),
            'c' => new MockRefObj(2, 5),
            'd' => new MockRefObj(3, 5),
            'e' => new MockRefObj(4, 5),
            'f' => new MockRefObj(5, 5),
        ]);
        $t2 = $lst->select(function (MockRefObj $item) {
            return $item->propA;
        })->toArray(true);
        $this->assertEquals(
            ['a' => 0, 'b' => 1, 'c' => 2, 'd' => 3, 'e' => 4, 'f' => 5],
            $t2
        );
        $t3 = $lst->select(function (MockRefObj $item) {
            return $item->propB;
        })->toArray(true);
        $this->assertEquals(
            ['a' => 5, 'b' => 5, 'c' => 5, 'd' => 5, 'e' => 5, 'f' => 5],
            $t3
        );
    }

    public function testEnumerableCount(): void
    {
        $lst = new Dictionary(['a' => 0, 'b' => 1, 'c' => 2, 'd' => 3, 'e' => 4, 'f' => 5, 'g' => 6, 'h' => 7]);
        $this->assertEquals(
            8,
            $lst->asEnumerable()->count()
        );
        $this->assertEquals(
            3,
            $lst->count(function ($item, $key) {
                return $item > 3 && $key != 'h';
            })
        );
    }

    public function testSum(): void
    {
        $lst = new Dictionary([
            'a' => 0, 'b' => 1, 'c' => 2, 'd' => 3, 'e' => 4, 'f' => 5, 'g' => 6, 'h' => 7
        ]);
        $t1 = $lst->sum();
        $this->assertEquals(
            28,
            $t1
        );

        $lst = new Dictionary([
            'a' => new MockRefObj(0, 5),
            'b' => new MockRefObj(1, 5),
            'c' => new MockRefObj(2, 5),
            'd' => new MockRefObj(3, 5),
            'e' => new MockRefObj(4, 5),
            'f' => new MockRefObj(5, 5),
        ]);
        $t2 = $lst->sum(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            15,
            $t2
        );
    }

    public function testAny(): void
    {
        $lst = new Dictionary([
            'a' => 0, 'b' => 1, 'c' => 2, 'd' => 3, 'e' => 4, 'f' => 5, 'g' => 6, 'h' => 7
        ]);

        $this->assertTrue(
            $lst->any()
        );
        $this->assertTrue(
            $lst->any(function ($item) {
                return $item > 2;
            })
        );
        $this->assertTrue(
            $lst->any(function ($item) {
                return $item === 0;
            })
        );
        $this->assertTrue(
            $lst->any(function ($item, $key) {
                return $item !== null && $key == 'b';
            })
        );
        $this->assertFalse(
            $lst->any(function ($item) {
                return $item > 15;
            })
        );
        $this->assertFalse(
            $lst->any(function ($item, $key) {
                return $item !== null && $key == 'zzz';
            })
        );
        $lst->clear();
        $this->assertFalse(
            $lst->any()
        );
    }

    public function testAll(): void
    {
        $lst = new Dictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        $this->assertTrue(
            $lst->all(function ($item) {
                return $item < 20;
            })
        );
        $this->assertTrue(
            $lst->all(function ($item) {
                return is_int($item);
            })
        );
        $this->assertFalse(
            $lst->all(function ($item) {
                return $item > 5;
            })
        );

        $lst->clear();
        $this->assertTrue(
            $lst->all(function ($item) {
                return $item > 5;
            })
        );
    }

    public function testAverage(): void
    {
        $lst = new Dictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->average();
        $this->assertEquals(
            5.0,
            $t1
        );

        $lst = new Dictionary([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $lst->average(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            2.5,
            $t2
        );
    }

    public function testMin(): void
    {
        $lst = new Dictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->min();
        $this->assertEquals(
            0,
            $t1
        );

        $lst = new Dictionary([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $lst->min(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            0,
            $t2
        );
        $lst->removeByKey(0);
        $lst->removeByKey(1);
        $t3 = $lst->min(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            2,
            $t3
        );
    }

    public function testMax(): void
    {
        $lst = new Dictionary([0, 1, 2, 3, 4, 5, 25, 6, 7, 8, 9, 10]);
        $t1 = $lst->max();
        $this->assertEquals(
            25,
            $t1
        );

        $lst = new Dictionary([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(25, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $lst->max(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            25,
            $t2
        );
    }

    public function testReverse(): void
    {
        $lst = new Dictionary([
            'a' => 0, 'b' => 1, 'c' => 2, 'd' => 3, 'e' => 4, 'f' => 5, 'g' => 6, 'h' => 7
        ]);
        $t1 = $lst->reverse();
        $this->assertEquals(
            [
                'h' => 7,
                'g' => 6,
                'f' => 5,
                'e' => 4,
                'd' => 3,
                'c' => 2,
                'b' => 1,
                'a' => 0,
            ],
            $t1->toArray(true)
        );
    }

    public function testAsEnumerable(): void
    {
        $lst = new Dictionary([
            'a' => 0, 'b' => 1, 'c' => 2, 'd' => 3, 'e' => 4, 'f' => 5, 'g' => 6, 'h' => 7
        ]);
        $t1 = $lst->where(function (int $item) {
            return $item > 3;
        });

        $t2 = $lst->where(function (int $item) {
            return $item > 3;
        })->asEnumerable();

        $lst->addKeyValue('m',8);

        $this->assertEquals(
            ['e' => 4, 'f' => 5, 'g' => 6, 'h' => 7, 'm' => 8],
            $t1->toArray(true)
        );
        $this->assertEquals(
            ['e' => 4, 'f' => 5, 'g' => 6, 'h' => 7],
            $t2->toArray(true)
        );
    }

    public function testSort(): void
    {
        $lst = new Dictionary([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->sort();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t1->toArray()
        );
        $t2 = $lst->sort(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t2->toArray()
        );
    }

    public function testSortDesc(): void
    {
        $lst = new Dictionary([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->sortDesc();
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t1->toArray()
        );
        $t2 = $lst->sortDesc(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t2->toArray()
        );
    }

    public function testOrderBy(): void
    {
        $lst = new Dictionary([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->orderBy();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t1->toArray()
        );
        $lst = new Dictionary([
            new MockRefObj(0, 3),
            new MockRefObj(1, 8),
            new MockRefObj(2, 1),
            new MockRefObj(3, 5),
            new MockRefObj(4, 6),
            new MockRefObj(5, 2),
        ]);
        $t2 = $lst->orderBy(function (MockRefObj $item) {
            return $item->propB;
        });
        $this->assertEquals(
            [
                new MockRefObj(2, 1),
                new MockRefObj(5, 2),
                new MockRefObj(0, 3),
                new MockRefObj(3, 5),
                new MockRefObj(4, 6),
                new MockRefObj(1, 8),
            ],
            $t2->toArray()
        );

        $t3 = $lst->orderBy(function (MockRefObj $item) {
            return $item->propB + $item->propA;
        });
        $this->assertEquals(
            [
                new MockRefObj(0, 3), // 3
                new MockRefObj(2, 1), // 3
                new MockRefObj(5, 2), // 7
                new MockRefObj(3, 5), // 8
                new MockRefObj(1, 8), // 9
                new MockRefObj(4, 6), // 10

            ],
            $t3->toArray()
        );

    }

    public function testOrderByDesc(): void
    {
        $lst = new Dictionary([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->orderByDesc();
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t1->toArray()
        );
        $lst = new Dictionary([
            new MockRefObj(0, 3),
            new MockRefObj(1, 8),
            new MockRefObj(2, 1),
            new MockRefObj(3, 5),
            new MockRefObj(4, 6),
            new MockRefObj(5, 2),
        ]);
        $t2 = $lst->orderByDesc(function (MockRefObj $item) {
            return $item->propB;
        });
        $this->assertEquals(
            [
                new MockRefObj(1, 8),
                new MockRefObj(4, 6),
                new MockRefObj(3, 5),
                new MockRefObj(0, 3),
                new MockRefObj(5, 2),
                new MockRefObj(2, 1),
            ],
            $t2->toArray()
        );

        $t3 = $lst->orderByDesc(function (MockRefObj $item) {
            return $item->propB + $item->propA;
        });
        $this->assertEquals(
            [
                new MockRefObj(4, 6), // 10
                new MockRefObj(1, 8), // 9
                new MockRefObj(3, 5), // 8
                new MockRefObj(5, 2), // 7
                new MockRefObj(2, 1), // 3
                new MockRefObj(0, 3), // 3
            ],
            $t3->toArray()
        );
    }

    public function testConcat(): void
    {
        $lst = new Dictionary([0, 1, 2, 3, 4, 5]);
        $t1 = $lst->concat([6, 7, 8, 9, 10]);
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
            $t1->toArray()
        );
    }

    public function testDistinct(): void
    {
        $lst = new Dictionary([0, 1, 0, 1, 1, 0]);
        $t1 = $lst->distinct();
        $this->assertEquals(
            [0, 1],
            $t1->toArray()
        );

        $lst = new Dictionary([
            new MockRefObj(1, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 4),
            new MockRefObj(3, 6),
            new MockRefObj(2, 4),

        ]);
        $t2 = $lst->distinct();
        $this->assertEquals(
            [
                new MockRefObj(1, 5),
                new MockRefObj(2, 4),
                new MockRefObj(3, 6),
            ],
            $t2->toArray()
        );

        $a = new MockRefObj(1, 5);
        $b = new MockRefObj(1, 5);
        $c = new MockRefObj(1, 5);

        $lst = new Dictionary([
            new MockRefObj(1, 5),
            $c,
            new MockRefObj(1, 5),
            $a,
            $b,
        ]);
        $t3 = $lst->distinct();
        $this->assertEquals(
            [
                new MockRefObj(1, 5),
            ],
            $t3->toArray()
        );

        $lst = new Dictionary([
            $a,
            $c,
            $b,
            $a,
            $b,
            $c,
        ]);
        $t4 = $lst->distinct(function ($a, $b) {
            return $a === $b;
        });
        $this->assertEquals(
            [
                $a,
                $c,
                $b,
            ],
            $t4->toArray()
        );
    }

    public function testExcept(): void
    {
        $lst = new Dictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->except([10, 9, 5, 6, 7, 8]);
        $this->assertEquals(
            [0, 1, 2, 3, 4],
            $t1->toArray()
        );
    }

    public function testFirstOrNUll(): void
    {
        $lst = new Dictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $this->assertEquals(
            0,
            $lst->firstOrNull()
        );
        $this->assertEquals(
            6,
            $lst->firstOrNull(function ($item) {
                return $item > 5;
            })
        );
        $this->assertEquals(
            null,
            $lst->firstOrNull(function ($item) {
                return $item > 15;
            })
        );
    }

    public function testAggregate(): void
    {
        $lst = new Dictionary([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $this->assertEquals(
            3628800,
            $lst->aggregate(function (&$total, $item) {
                $total *= $item;
            })
        );
        $this->assertEquals(
            181440,
            $lst->aggregate(
                function (&$total, $item) {
                    $total *= $item;
                },
                null,
                function ($total, Enumerable $th) {
                    return ($total / $th->count()) / 2;
                })
        );
        $this->assertEquals(
            3.25,
            $lst->aggregate(
                function (&$total, $item) {
                    $total += $item;
                },
                10,
                function ($total, Enumerable $th) {
                    return ($total / $th->count()) / 2;
                })
        );
    }

    public function testUnion(): void
    {
        $lst = new Dictionary([-3, '1', 1, 2, 3, 3, 4, 4, 5]);
        $t1 = $lst->union([3, 8, 4, 8]);

        $this->assertEquals(
            [-3, 1, 2, 3, 4, 5, 8],
            $t1->toArray()
        );
        $t2 = $lst->union([3, 8, 4, 8], function ($a, $b) {
            return $a === $b;
        });
        $this->assertEquals(
            [-3, '1', 1, 2, 3, 4, 5, 8],
            $t2->toArray()
        );
    }

    public function testIntersect(): void
    {
        $lst = new Dictionary([1, 3, 4, 5]);
        $t1 = $lst->intersect([3, 8, 4]);

        $this->assertEquals(
            [3, 4],
            $t1->toArray()
        );
        $t2 = $lst->intersect([]);

        $this->assertEquals(
            [],
            $t2->toArray()
        );

        $a = new MockRefObj(1, 5);
        $b = new MockRefObj(1, 5);
        $c = new MockRefObj(1, 5);
        $d = new MockRefObj(1, 5);
        $e = new MockRefObj(1, 5);
        $f = new MockRefObj(1, 5);

        $lst2 = new Dictionary([$a, $b, $c, $d]);
        $t3 = $lst2->intersect([$f, $d, $e, $b], function ($first, $second) {
            return $first === $second;
        });

        $this->assertEquals(
            [$b, $d],
            $t3->toArray()
        );

    }

    public function testZip(): void
    {
        $lst = new Dictionary([1, 2, 3, 4, 5]);
        $t1 = $lst->zip(new Dictionary(['a', 'b', 'c', 'd', 'e']), function ($f, $s) {
            return $f . '-' . $s;
        });

        $this->assertEquals(
            ['1-a', '2-b', '3-c', '4-d', '5-e'],
            $t1->toArray()
        );

        $t2 = $lst->zip(new Dictionary(['a', 'b', 'c']), function ($f, $s) {
            return $f . '-' . $s;
        });

        $this->assertEquals(
            ['1-a', '2-b', '3-c'],
            $t2->toArray()
        );

        $lst2 = new Dictionary([1, 2]);
        $t3 = $lst2->zip(new Dictionary(['a', 'b', 'c', 'd', 'e']), function ($f, $s) {
            return $f . '-' . $s;
        });

        $this->assertEquals(
            ['1-a', '2-b'],
            $t3->toArray()
        );
    }

    public function testLastOrNull(): void
    {
        $enmr = new Dictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $enmr->lastOrNull();
        $this->assertEquals(
            10,
            $t1
        );
        $t2 = $enmr->lastOrNull(function ($item) {
            return $item < 5;
        });
        $this->assertEquals(
            4,
            $t2
        );
    }

    #endregion Enumerable

    #region serializable

    public function testSerializable(){
        $arr = [
            'a' => 0,
            'b' => 1,
            'c' => 2,
            'd' => 3,
            'e' => 4,
            'f' => 5,
            'g' => 6,
            'h' => 7,
        ];
        $lst = new Dictionary($arr);
        $ser = serialize($lst);
        $this->assertNotEmpty($ser);
        $this->assertIsString($ser);
        /** @var  Dictionary $unSer */
        $unSer = unserialize($ser);
        $this->assertTrue($unSer instanceof Dictionary);

        $this->assertEquals(
            $arr,
            $unSer->toArray(true)
        );
    }

    #endregion serializable


}
