<?php


namespace Gek\Collections\Iterators;


use ArrayIterator;
use Closure;
use Iterator;

/**
 * Class SortIterator
 * @package Gek\Collections\Iterators
 */
class SortIterator implements Iterator
{
    #region fields

    protected Iterator $iterator;

    protected ?Iterator $sortedIterator = null;

    protected Closure $compareFn;

    protected bool $desc = false;

    protected bool $keySort = false;

    #endregion fields

    #region ctor
    /**
     * SortIterator constructor.
     * @param Iterator $iterator
     * @param callable $compareFn
     * @param bool $desc
     * @param bool $keySort
     */
    public function __construct(Iterator $iterator,callable $compareFn, bool $desc = false, bool $keySort = false){
        $this->iterator = $iterator;
        $this->compareFn = ($compareFn instanceof Closure) ? $compareFn : Closure::fromCallable($compareFn);
        $this->desc = $desc;
        $this->keySort = $keySort;
    }

    #endregion ctor

    #region Iterator

    /**
     * Return the current element
     * @link https://php.net/manual/en/iterator.current.php
     * @return mixed Can return any type.
     * @since 5.0.0
     */
    public function current()
    {
        $this->check();
        return $this->sortedIterator->current();
    }

    /**
     * Move forward to next element
     * @link https://php.net/manual/en/iterator.next.php
     * @return void Any returned value is ignored.
     * @since 5.0.0
     */
    public function next()
    {
        $this->check();
        $this->sortedIterator->next();
    }

    /**
     * Return the key of the current element
     * @link https://php.net/manual/en/iterator.key.php
     * @return mixed scalar on success, or null on failure.
     * @since 5.0.0
     */
    public function key()
    {
        $this->check();
        return $this->sortedIterator->key();
    }

    /**
     * Checks if current position is valid
     * @link https://php.net/manual/en/iterator.valid.php
     * @return bool The return value will be casted to boolean and then evaluated.
     * Returns true on success or false on failure.
     * @since 5.0.0
     */
    public function valid()
    {
        $this->check();
        return $this->sortedIterator->valid();
    }

    /**
     * Rewind the Iterator to the first element
     * @link https://php.net/manual/en/iterator.rewind.php
     * @return void Any returned value is ignored.
     * @since 5.0.0
     */
    public function rewind()
    {
        $this->sortedIterator = null;
        $this->iterator->rewind();
    }

    #endregion Iterator

    #region utils

    protected function check(){
        if($this->sortedIterator !== null){
            return;
        }
        $tempArr = iterator_to_array($this->iterator);
        if($this->keySort){
            uksort($tempArr,$this->compareFn);
        }else{
            uasort($tempArr,$this->compareFn);
        }
        if($this->desc){
            $tempArr = array_reverse($tempArr);
        }
        $this->sortedIterator = new ArrayIterator($tempArr);
    }

    #endregion utils

}
