<?php


namespace Gek\Collections;


use Countable;
use Gek\Collections\Iterators\FixedArrayListIterator;
use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Gek\Infrastructure\Str;
use Iterator;
use IteratorAggregate;
use SplFixedArray;
use Traversable;

/**
 * Class FixedArrayList
 * @package Gek\Collections
 */
class FixedArrayList extends Enumerable implements IList
{
    #region fields

    protected SplFixedArray $splFixedArray;

    protected int $defaultCapacity = 4;

    protected int $listCount = 0;

    protected int $capacity = 0;

    #endregion fields

    #region ctor

    /**
     * FixedArrayList constructor.
     * @param Traversable|IEnumerable|int|null $capacityOrEnumerable
     */
    public function __construct($capacityOrEnumerable = null)
    {
        if ($capacityOrEnumerable === null) {
            $this->splFixedArray = new SplFixedArray(0);
        } else {
            if (is_int($capacityOrEnumerable)) {
                $this->capacity = $capacityOrEnumerable;
                $this->splFixedArray = new SplFixedArray($capacityOrEnumerable);
            } elseif (is_array($capacityOrEnumerable)) {
                $this->splFixedArray = SplFixedArray::fromArray($capacityOrEnumerable, false);
                $this->listCount = $this->splFixedArray->count();
                $this->capacity = $this->listCount;
            } elseif ($capacityOrEnumerable instanceof Enumerable) {
                $this->splFixedArray = $capacityOrEnumerable->toSplFixedArray();
                $this->listCount = $this->splFixedArray->count();
                $this->capacity = $this->listCount;
            } elseif ($capacityOrEnumerable instanceof IteratorAggregate) {
                $this->splFixedArray = SplFixedArray::fromArray(iterator_to_array($capacityOrEnumerable->getIterator()), false);
                $this->listCount = $this->splFixedArray->count();
                $this->capacity = $this->listCount;
            } elseif ($capacityOrEnumerable instanceof Iterator) {
                $this->splFixedArray = SplFixedArray::fromArray(iterator_to_array($capacityOrEnumerable), false);
                $this->listCount = $this->splFixedArray->count();
                $this->capacity = $this->listCount;
            } elseif ($capacityOrEnumerable instanceof Traversable) {
                $tmpArr = array();
                $count = 0;
                foreach ($capacityOrEnumerable as $itm) {
                    $tmpArr[] = $itm;
                    $count++;
                }
                $this->splFixedArray = SplFixedArray::fromArray($tmpArr);
                $this->listCount = $count;
                $this->capacity = $this->listCount;
            }elseif (is_object($capacityOrEnumerable)){
                $tmpArr = array();
                $count = 0;
                foreach ($capacityOrEnumerable as $itm) {
                    $tmpArr[] = $itm;
                    $count++;
                }
                $this->splFixedArray = SplFixedArray::fromArray($tmpArr);
                $this->listCount = $count;
                $this->capacity = $this->listCount;
            }
        }
        parent::__construct(new FixedArrayListIterator($this));
    }

    #endregion ctor

    #region IList

    /**
     * Whether a offset exists
     * @link https://php.net/manual/en/arrayaccess.offsetexists.php
     * @param mixed $offset <p>
     * An offset to check for.
     * </p>
     * @return bool true on success or false on failure.
     * </p>
     * <p>
     * The return value will be casted to boolean if non-boolean was returned.
     * @since 5.0.0
     */
    public function offsetExists($offset)
    {
        if ($offset < 0 || $offset >= $this->listCount) {
            return false;
        }
        return true;
    }

    /**
     * Offset to retrieve
     * @link https://php.net/manual/en/arrayaccess.offsetget.php
     * @param mixed $offset <p>
     * The offset to retrieve.
     * </p>
     * @return mixed Can return all value types.
     * @since 5.0.0
     */
    public function offsetGet($offset)
    {
        if ($offset < 0 || $offset >= $this->listCount) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz ofset : {0}', $offset),
                0,
                null,
                '$offset'
            );
        }
        return $this->splFixedArray->offsetGet($offset);
    }

    /**
     * Offset to set
     * @link https://php.net/manual/en/arrayaccess.offsetset.php
     * @param mixed $offset <p>
     * The offset to assign the value to.
     * </p>
     * @param mixed $value <p>
     * The value to set.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetSet($offset, $value)
    {

        if ($offset < 0 || $offset >= $this->listCount) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz ofset : {0}', $offset),
                0,
                null,
                '$offset'
            );
        }
        $this->splFixedArray->offsetSet($offset, $value);
    }

    /**
     * Offset to unset
     * @link https://php.net/manual/en/arrayaccess.offsetunset.php
     * @param mixed $offset <p>
     * The offset to unset.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetUnset($offset)
    {
        if ($offset < 0 || $offset >= $this->listCount) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz ofset : {0}', $offset),
                0,
                null,
                '$offset'
            );
        }
        $this->splFixedArray->offsetUnset($offset);
        FixedArrayHelper::floatLeft($this->splFixedArray, $offset, 1);
        $this->listCount--;
    }

    /**
     * @param mixed $item
     */
    public function add($item): void
    {
        $this->ensureCapacity($this->listCount + 1);
        $this->splFixedArray[$this->listCount] = $item;
        $this->listCount++;
    }

    /**
     * @param array|Enumerable|\IteratorAggregate|\Iterator|\Traversable $items
     */
    public function addRange($items): void
    {
       $this->insertRange($this->listCount,$items);
    }

    /**
     *
     */
    public function clear(): void
    {
        $this->splFixedArray = new SplFixedArray();
        $this->listCount = 0;
        $this->capacity = 0;
    }

    /**
     * {@inheritDoc}
     */
    public function contains($item): bool
    {
        for ($i = 0; $i < $this->listCount; $i++){
            $fullEqual = !is_object($item) || !is_object($this->splFixedArray[$i]);
            $res = $fullEqual ?
                ($this->splFixedArray[$i] === $item):
                ($this->splFixedArray[$i] == $item);
            if($res){
                return true;
            }
        }
        return false;
    }

    /**
     * @param mixed $item
     * @return bool
     */
    public function remove($item): bool
    {
        $index = $this->indexOf($item);
        if ($index == -1) {
            return false;
        }

        $this->removeAt($index);
        return true;
    }

    /**
     * @param mixed $item
     * @return int
     */
    public function indexOf($item): int
    {
        for ($i = 0; $i < $this->listCount; $i++) {

            if ($this->splFixedArray[$i] == $item) {
                return $i;
            }
        }
        return -1;
    }

    /**
     * @param int $index
     * @param mixed $item
     */
    public function insert(int $index, $item): void
    {
        if ($index < 0 || $index > $this->listCount) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz index : {0}', $index),
                0,
                null,
                '$index'
            );
        }
        $this->ensureCapacity($this->listCount + 1);
        FixedArrayHelper::floatRight($this->splFixedArray, $index, 1, $this->listCount + 1);
        $this->splFixedArray[$index] = $item;
        $this->listCount++;
    }

    /**
     * @param int $index
     * @param array|Enumerable|\IteratorAggregate|\Iterator|\Traversable $items
     */
    public function insertRange(int $index, $items): void
    {
        if ($index < 0 || $index > $this->listCount) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz index : {0}', $index),
                0,
                null,
                '$index'
            );
        }
        if (!is_array($items) && !($items instanceof Countable)) {
            if ($items instanceof IteratorAggregate) {
                $items = iterator_to_array($items->getIterator());
            } elseif ($items instanceof Iterator) {
                $items = iterator_to_array($items);
            } elseif (($items instanceof Traversable) || is_object($items)) {
                $temp = array();
                foreach ($items as $itm){
                    $temp[] = $itm;
                }
                $items = $temp;
                unset($temp);
            }
        }

        $itemsCount = count($items);
        $this->ensureCapacity($this->listCount + $itemsCount);
        if($index < $this->listCount){
            FixedArrayHelper::floatRight($this->splFixedArray,$index,$itemsCount,$this->listCount + $itemsCount);
        }
        $i = $index;
        foreach ($items as $item){
            $this->splFixedArray[$i] = $item;
            $i++;
        }
        $this->listCount += $itemsCount;
    }

    /**
     * @param int $index
     */
    public function removeAt(int $index): void
    {
        if ($index < 0 || $index >= $this->listCount) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz index : {0}', $index),
                0,
                null,
                '$index'
            );
        }
        unset($this->splFixedArray[$index]);
        FixedArrayHelper::floatLeft($this->splFixedArray, $index, 1);
        $this->listCount--;
    }

    /**
     * @param int $index
     * @param int $count
     */
    public function removeRange(int $index, int $count): void
    {
        if ($index < 0 || $index >= $this->listCount) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz index : {0}', $index),
                0,
                null,
                '$index'
            );
        }
        $lastIndex = ($index + $count) - 1;
        if ($lastIndex < 0 || $lastIndex >= $this->listCount) {
            throw new ArgumentOutOfRangeException(
                Str::format('Geçersiz count : {0}', $count),
                0,
                null,
                '$count'
            );
        }
        FixedArrayHelper::floatLeft($this->splFixedArray,$index,$count);
        $this->listCount -= $count;

    }

    /**
     * @param callable $fn
     * @return int
     */
    public function removeAll(callable $fn): int
    {
        $rmvItems = $this->where($fn)->toArray();
        $removedCount = 0;
        foreach ($rmvItems as $key => $val){
            $this->remove($val);
            $removedCount++;
        }
        unset($rmvItems);
        return $removedCount;
    }

    /**
     * @param callable|null $fn
     * @return int
     */
    public function count(?callable $fn = null): int
    {
        if ($fn === null) {
            return $this->listCount;
        }
        return parent::count($fn);
    }

    #endregion IList

    #region Enumerable Overrides
    /**
     * @param callable|null $fn
     * @return mixed|null
     */
    public function firstOrNull(?callable $fn = null)
    {
        if($this->listCount == 0){
            return null;
        }
        if($fn === null){
            return $this[0];
        }
        return parent::firstOrNull($fn);
    }

    /**
     * @param callable|null $fn
     * @return mixed|null
     */
    public function lastOrNull(?callable $fn = null)
    {
        if($this->listCount == 0){
            return null;
        }
        if($fn === null){
            return $this[$this->listCount - 1];
        }
        for ($i = $this->listCount - 1; $i >= 0; $i--){
            if($fn($this[$i],$i,$this->iterator) === true){
                return $this[$i];
            }
        }
        return null;
    }

    #endregion Enumerable Overrides

    /**
     * @return int
     */
    public function getCapacity():int {
        return $this->capacity;
    }

    #region utils

    /**
     * @param int $minCapacity
     */
    protected function ensureCapacity(int $minCapacity): void
    {

        if ($this->capacity < $minCapacity) {
            $newCapacity = $this->capacity == 0 ? $this->defaultCapacity : ($this->capacity * 2);
            if ($newCapacity < $minCapacity) {
                $newCapacity = $minCapacity;
            }
            // $cnt++;
            // echo Str::format('SetSize {0} : {1} {2}', $cnt,$newCapacity,PHP_EOL);
            $this->splFixedArray->setSize($newCapacity);
            $this->capacity = $newCapacity;
        }
    }

    #endregion utils

}
